<?php

namespace App\Http\Controllers;

use App\Models\AccountBroughtForward;
use App\Models\AccountingYear;
use App\Models\AccountType;
use App\Models\Expense;
use App\Models\Income;
use App\Models\Menu;
use App\Models\Organisation;
use App\Services\PermissionService;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Traits\CommonTrait;
use Illuminate\Database\QueryException;
use Illuminate\Support\Facades\Auth;

class CapitalController extends Controller
{
    protected $permissionService;
    protected $menuId;
    protected $currentYear;
    protected $currentOrgId;
    protected $currentJournalTransactionsTable;
    use  CommonTrait;

    public function __construct(PermissionService $permissionService)
    {
        $this->permissionService = $permissionService;
        $this->currentJournalTransactionsTable = $this->getYearPrefixedTables()['journal_transactions'];
        $this->menuId = Menu::where('route', 'capital.index')->value('id');
        $this->currentYear = AccountingYear::current();
        $this->currentOrgId =  Auth::user()->organisation_id;
    }

    public function index()
    {

        $today = Carbon::now();
        $year = $today->year;

        if ($today->month < 4) {
            $startYear = $year - 1;
            $endYear   = $year;
        } else {
            $startYear = $year;
            $endYear   = $year + 1;
        }

        $financialYearStart = Carbon::createFromDate($startYear, 4, 1)->toDateString();

        // return $financialYearStart;
        $recordExists = AccountBroughtForward::where('dated_as_on', $financialYearStart)->count();
        // return $recordExists;
        return view('auth.capitals.index', compact('recordExists'));
    }



    public function getcapitalDetails(Request $request)
    {
        if (!$this->permissionService->hasPermission($this->menuId, 'r')) {
            abort(403, 'You do not have read access to Capital Records.');
        }

        $search = $request->input('search');
        $page   = max(1, (int) $request->input('page', 1));
        $size   = max(1, (int) $request->input('size', 10));

        $sortField = $request->input('sorters.0.field', 'id');
        $sortOrder = $request->input('sorters.0.dir', 'desc');

        $result = AccountBroughtForward::with('financialYear')
            ->when(
                $search,
                fn($q) =>
                $q->where(function ($query) use ($search) {
                    $query->where('cash_in_hand', 'like', "%{$search}%")
                        ->orWhere('cash_at_bank', 'like', "%{$search}%")
                        ->orWhereHas(
                            'financialYear',
                            fn($q) =>
                            $q->where('accounting_year_financial', 'like', "%{$search}%")
                        );
                })
            )
            ->orderBy($sortField, $sortOrder)
            ->paginate($size, ['*'], '', $page);


        return response()->json([
            'data'         => $result->items(),
            'current_page' => $result->currentPage(),
            'last_page'    => $result->lastPage(),
            'per_page'     => $result->perPage(),
            'total'        => $result->total(),
        ]);
    }

    public function create()
    {
        $years = AccountingYear::where('status', 'active')->get();
        return view('auth.capitals.create', compact('years'));
    }

    public function store(Request $request)
    {
        if (!$this->permissionService->hasPermission($this->menuId, 'w')) {
            abort(403, 'You do not have write access to Capital Records.');
        }

        $validated = $request->validate([
            'accounting_year_id' => 'required',
            'cash_in_hand'       => 'required',
            'cash_at_bank'       => 'required',
            'transaction_date'   => 'required',
        ]);
        if (!$this->currentOrgId) {
            return redirect()->back()
                ->withInput()
                ->withErrors(
                    ['organisation_id' => 'Please select a current organisation before adding user.'],
                    'orgCheck'
                );
        }
        try {
            AccountBroughtForward::create([
                'financial_year_id' => $validated['accounting_year_id'],
                'cash_in_hand'      => $validated['cash_in_hand'],
                'cash_at_bank'      => $validated['cash_at_bank'],
                'user_id'           => auth()->user()->id,
                'dated_as_on'       => $validated['transaction_date'],
                'organisation_id'   => $this->currentOrgId,
            ]);

            return redirect()->route('capital.index')->with('success', 'Capital added successfully.');
        } catch (QueryException $e) {
            return redirect()->back()
                ->withInput()
                ->withErrors([
                    'organisation_id' => 'Please select a current organisation before adding capital. From Accounting Years add section.'
                ]);
        }
    }



    public function show(string $id)
    {
        //
    }


    public function edit(string $id)
    {
        if (!$this->permissionService->hasPermission($this->menuId, 'x')) {
            abort(403, 'You do not have execute access to Capital Records.');
        }

        $id = base64_decode($id);
        // return $id;
        $accountCapital = AccountBroughtForward::where('id', $id)->first();
        $years = AccountingYear::where('status', 'active')->get();

        return view('auth.capitals.edit', compact('accountCapital', 'years'));
    }


    public function update(Request $request, string $id)
    {
        // return $request;
        if (!$this->permissionService->hasPermission($this->menuId, 'x')) {
            abort(403, 'You do not have write access to Capital Records.');
        }

        $validated = $request->validate([
            'accounting_year_id' => 'required',
            'cash_in_hand'          => 'required',
            'cash_at_bank'  => 'required',
            'transaction_date' => 'required',
        ]);

        $accountCapital = AccountBroughtForward::findOrFail($id);

        $accountCapital->update([
            'financial_year_id' => $validated['accounting_year_id'] ?? $accountCapital->financial_year_id,
            'cash_in_hand' => $validated['cash_in_hand'] ?? $accountCapital->cash_in_hand,
            'cash_at_bank'    => $validated['cash_at_bank'] ?? $accountCapital->cash_at_bank,
            'user_id'   => auth()->user()->id ?? $accountCapital->user_id,
            'dated_as_on' => $validated['transaction_date'] ?? $accountCapital->dated_as_on
        ]);
        return redirect()->route('capital.index')->with('success', 'Capital updated successfully.');
    }

    public function destroy(string $id)
    {
        //
    }

    public function netCapital()
    {
        $getCashAndBank = $this->cashAndBankSumation();
        return view('capitals.total', ['totalCash' => $getCashAndBank['total_cash_in_hand'], 'totalBank'  => $getCashAndBank['total_cash_at_bank']]);
    }
    public function getcapitalAmountDetails($flag)
    {
        if ($flag != 'cash' && $flag != 'bank') {
            abort(404, 'Page Not Found');
        }

        $paymentType = $flag === 'cash' ? '=' : '>';

        $totals = [
            'income'  => 0,
            'expense' => 0,
        ];

        $transactions = DB::table($this->currentJournalTransactionsTable . ' as jt')
            ->leftJoin('donors as d', 'd.donor_id', '=', 'jt.donor_id')
            ->leftJoin('vendors as v', 'v.vendor_id', '=', 'jt.vendor_id')
            ->leftJoin('heads as h', 'h.id', '=', 'jt.head_id')
            ->leftJoin('payment_types as pt', 'pt.id', '=', 'jt.payment_type_id')
            ->where('jt.organisation_id', $this->currentOrgId)
            ->where('jt.payment_type_id', $paymentType, 1)
            ->whereNull('jt.deleted_at')
            ->select(
                'd.donor_name',
                'd.donor_type',
                'v.vendor_name',
                'v.vendor_type',
                'h.name as head_name',
                'jt.transaction_amount',
                'jt.transaction_date',
                'jt.transaction_narration',
                'jt.transcation_type'
            )
            ->orderBy('jt.transaction_date','desc')
            ->get();

        foreach ($transactions as $row) {
            if ($row->transcation_type === 'income') {
                $totals['income'] += $row->transaction_amount;
            } elseif ($row->transcation_type === 'expense') {
                $totals['expense'] += $row->transaction_amount;
            }
        }
        $accountBroughtForward = AccountBroughtForward::where('financial_year_id', $this->currentYear->accounting_year_id)->first();
        return view('capitals.totalHistory', [
            'transactions'      => $transactions,
            'totals'            => $totals,
            'flag'              => $flag,
            'brought_forward'   => $flag === 'bank' ? $accountBroughtForward->cash_at_bank : $accountBroughtForward->cash_in_hand
        ]);
    }
}
