<?php

namespace App\Http\Controllers;

use App\Models\bank;
use App\Models\Head;
use App\Models\Organisation;
use App\Models\Bank as BankModel;
use App\Traits\CommonTrait;
use App\Http\Requests\StoreContraTransactionRequest;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class ContraController extends Controller
{
    use CommonTrait;

    protected array $tables = [];
    protected ?string $journalTable = null;

    public function __construct()
    {
        $this->tables = $this->getYearPrefixedTables();
        $this->journalTable = $this->tables['journal_transactions'] ?? null;
    }

    /**
     * Show the contra transaction entry screen.
     */
    public function create()
    {
        if (! $this->journalTable) {
            return redirect()
                ->route('dashboard')
                ->with('error', 'Please configure the active financial year before recording contra transactions.');
        }

        $organisations = Organisation::where('status', 'active')
            ->orderBy('organisation_name')
            ->get(['organisation_id', 'organisation_name', 'is_current']);

        $currentOrganisation = Organisation::current();

        $heads = Head::where('status', 'active')
            ->orderBy('name')
            ->get(['id', 'name', 'type', 'head_group_id', 'account_type_id', 'organisation_id']);


        $banks = $this->getActiveBanks();
        return view('jounaltransactions.contra.add-contra-transaction', [
            'organisations' => $organisations,
            'currentOrganisation' => $currentOrganisation,
            'banks' => $banks,
        ]);
    }

    /**
     * Store a newly created contra transaction.
     *
     * @param  \App\Http\Requests\StoreContraTransactionRequest  $request
     * @return \Illuminate\Http\JsonResponse|\Illuminate\Http\RedirectResponse
     */
    public function store(StoreContraTransactionRequest $request)
    {
        if (! $this->journalTable) {
            return $request->ajax()
                ? response()->json(['success' => false, 'message' => 'Active financial year not found.'], 400)
                : back()->with('error', 'Active financial year not found.');
        }

        // ----------------------------
        // START TRANSACTION
        // ----------------------------
        DB::beginTransaction();

        try {
            // Get validated data
            $validated = $request->validated();

            // ========= SINGLE ENTRY IN JOURNAL TABLE ==========
            $journalData = [
                'organisation_id'       => $validated['organisation_id'],
                'transaction_reference' => $validated['transaction_reference'] ?? null,
                'transcation_type'      => 'contra',
                'user_id'               => Auth::id(),
                'contra_reference_type' => $validated['type'],
                'transaction_amount'    => $validated['transaction_amount'],
                'bank_id'               => $validated['bank_id'] ?? null,
                'from_bank_id'          => ($validated['type'] === 'bank_transfer') ? ($validated['from_bank_id'] ?? null) : null,
                'transaction_date'      => $validated['transaction_date'] ?? now(),
                'transaction_narration' => $validated['transaction_narration'] ?? null,
                'created_at'            => now(),
            ];

            // Create single journal entry
            DB::table($this->journalTable)->insert($journalData);

            DB::commit();

            // SUCCESS RESPONSE
            if ($request->ajax()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Contra transaction saved successfully.'
                ]);
            }

            return back()->with('success', 'Contra transaction saved successfully.');
        } catch (\Throwable $th) {
            DB::rollBack();
            report($th);

            return $request->ajax()
                ? response()->json(['success' => false, 'message' => $th->getMessage()], 500)
                : back()->with('error', 'Failed to record contra transaction.');
        }
    }

    /**
     * Get contra transaction details for AJAX/Tabulator.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getContraDetails(Request $request)
    {
        if (! $this->journalTable) {
            return response()->json(['error' => 'Active financial year not found.'], 400);
        }

        $user = Auth::user();
        $search = $request->input('search');
        $organisation_id = $request->input('organisation_id');
        $page = max(1, (int) $request->input('page', 1));
        $size = max(1, (int) $request->input('size', 10));
        $sortField = $request->input('sorters.0.field', 'id');
        $sortOrder = $request->input('sorters.0.dir', 'desc');

        $query = DB::table($this->journalTable)
            ->where('transcation_type', 'contra')
            ->when($user && $user->id !== 1, function ($q) use ($user) {
                $q->where('organisation_id', $user->organisation_id);
            })
            ->when($organisation_id, function ($q) use ($organisation_id) {
                $q->where('organisation_id', $organisation_id);
            })
            ->when($search, function ($q) use ($search) {
                $q->where(function ($query) use ($search) {
                    $query->where('transaction_reference', 'like', "%{$search}%")
                        ->orWhere('transaction_amount', 'like', "%{$search}%")
                        ->orWhere('transaction_narration', 'like', "%{$search}%")
                        ->orWhere('contra_reference_type', 'like', "%{$search}%");
                });
            });

        $total = $query->count();
        $offset = ($page - 1) * $size;

        $contraJournals = $query->orderBy($sortField, $sortOrder)
            ->offset($offset)
            ->limit($size)
            ->get();

        // Get organisations and banks
        $orgIds = $contraJournals->pluck('organisation_id')->unique()->filter();
        $organisations = Organisation::whereIn('organisation_id', $orgIds)->get()->keyBy('organisation_id');

        $bankIds = $contraJournals->pluck('bank_id')->merge(
            $contraJournals->pluck('from_bank_id')
        )->unique()->filter();
        $banks = BankModel::whereIn('id', $bankIds)->get()->keyBy('id');

        // Format data
        $data = $contraJournals->map(function ($journal) use ($organisations, $banks) {
            $bank = $banks->get($journal->bank_id);
            $fromBank = $journal->from_bank_id ? $banks->get($journal->from_bank_id) : null;
            
            $bankName = '';
            if ($fromBank && $bank) {
                $bankName = $fromBank->name . ' → ' . $bank->name;
            } elseif ($bank) {
                $bankName = $bank->name;
            }elseif($fromBank){
                $bankName = $fromBank->name;
            }

            $org = $organisations->get($journal->organisation_id);

            return [
                'id' => $journal->id,
                'transaction_reference' => $journal->transaction_reference ?? null,
                'contra_reference_type' => $journal->contra_reference_type ?? null,
                'transaction_amount' => $journal->transaction_amount ?? 0,
                'transaction_narration' => $journal->transaction_narration ?? null,
                'transaction_date' => $journal->transaction_date ?? $journal->created_at,
                'created_at' => $journal->created_at,
                'organisation' => $org ? [
                    'organisation_id' => $org->organisation_id,
                    'organisation_name' => $org->organisation_name,
                ] : null,
                'organisation_name' => $org ? $org->organisation_name : null,
                'bank_names' => $bankName,
            ];
        });

        return response()->json([
            'data' => $data->values(),
            'current_page' => $page,
            'last_page' => ceil($total / $size),
            'per_page' => $size,
            'total' => $total,
        ]);
    }

    /**
     * Display a listing of contra transactions.
     *
     * @return \Illuminate\View\View
     */
    public function index()
    {
        if (! $this->journalTable) {
            return redirect()
                ->route('dashboard')
                ->with('error', 'Please configure the active financial year before viewing contra transactions.');
        }

        $user = Auth::user();
        $organisations = Organisation::where('status', 'active')
            ->orderBy('organisation_name')
            ->get(['organisation_id', 'organisation_name', 'is_current']);

        // Get cash and bank calculations
        $cashBankData = $this->cashAndBankSumation();

        return view('jounaltransactions.contra.index', [
            'organisations' => $organisations,
            'cashBankData' => $cashBankData,
        ]);
    }

    /**
     * Display the specified contra transaction.
     *
     * @param  int  $id
     * @return \Illuminate\View\View|\Illuminate\Http\RedirectResponse
     */
    public function show($id)
    {
        if (! $this->journalTable) {
            return redirect()
                ->route('transactions.contra.index')
                ->with('error', 'Active financial year not found.');
        }

        $user = Auth::user();

        // Get the contra journal entry
        $contraJournal = DB::table($this->journalTable)
            ->where('id', $id)
            ->where('transcation_type', 'contra')
            ->when($user && $user->id !== 1, function ($q) use ($user) {
                $q->where('organisation_id', $user->organisation_id);
            })
            ->first();

        if (!$contraJournal) {
            return redirect()
                ->route('transactions.contra.index')
                ->with('error', 'Contra transaction not found.');
        }

        // Get bank details
        $bankIds = collect([$contraJournal->bank_id, $contraJournal->from_bank_id])->filter()->unique();
        $banks = BankModel::whereIn('id', $bankIds)->get()->keyBy('id');

        // Get organisation
        $organisation = Organisation::find($contraJournal->organisation_id);

        // Get cash and bank calculations
        $cashBankData = $this->cashAndBankSumation();

        return view('jounaltransactions.contra.show', [
            'contraJournal' => $contraJournal,
            'banks' => $banks,
            'organisation' => $organisation,
            'cashBankData' => $cashBankData,
        ]);
    }

    /**
     * Get contra transaction details for modal (AJAX).
     *
     * @param  int  $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function getDetailsForModal($id)
    {
        if (! $this->journalTable) {
            return response()->json(['error' => 'Active financial year not found.'], 400);
        }

        $user = Auth::user();

        // Get the contra journal entry
        $contraJournal = DB::table($this->journalTable)
            ->where('id', $id)
            ->where('transcation_type', 'contra')
            ->when($user && $user->id !== 1, function ($q) use ($user) {
                $q->where('organisation_id', $user->organisation_id);
            })
            ->first();

        if (!$contraJournal) {
            return response()->json(['error' => 'Contra transaction not found.'], 404);
        }

        // Get bank details
        $bankIds = collect([$contraJournal->bank_id, $contraJournal->from_bank_id])->filter()->unique();
        $banks = BankModel::whereIn('id', $bankIds)->get()->keyBy('id');

        $bank = $banks->get($contraJournal->bank_id);
        $fromBank = $contraJournal->from_bank_id ? $banks->get($contraJournal->from_bank_id) : null;

        // Get organisation
        $organisation = Organisation::find($contraJournal->organisation_id);

        // Get cash and bank calculations
        $cashBankData = $this->cashAndBankSumation();

        return response()->json([
            'success' => true,
            'contraJournal' => [
                'id' => $contraJournal->id,
                'transaction_reference' => $contraJournal->transaction_reference ?? null,
                'contra_reference_type' => $contraJournal->contra_reference_type ?? null,
                'transaction_amount' => $contraJournal->transaction_amount ?? 0,
                'transaction_narration' => $contraJournal->transaction_narration ?? null,
                'transaction_date' => $contraJournal->transaction_date ?? null,
                'created_at' => $contraJournal->created_at,
            ],
            'organisation' => $organisation ? [
                'organisation_id' => $organisation->organisation_id,
                'organisation_name' => $organisation->organisation_name,
            ] : null,
            'bank' => $bank ? [
                'id' => $bank->id,
                'name' => $bank->name,
                'account_number' => $bank->account_number,
                'branch' => $bank->branch,
            ] : null,
            'from_bank' => $fromBank ? [
                'id' => $fromBank->id,
                'name' => $fromBank->name,
                'account_number' => $fromBank->account_number,
                'branch' => $fromBank->branch,
            ] : null,
            'cashBankData' => $cashBankData,
        ]);
    }
}
