<?php

namespace App\Http\Controllers;

use App\Http\Requests\StoreHeadGroupRequest;
use App\Models\HeadGroup;
use App\Models\Menu;
use App\Models\Organisation;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Http\RedirectResponse;
use Illuminate\Support\Facades\Validator;
use App\Services\PermissionService;
use App\Models\AccountingYear;


class HeadGroupController extends Controller
{
    protected $permissionService;
    protected $menuId;
    protected $currentYear;

    public function __construct(PermissionService $permissionService)
    {
        $this->permissionService = $permissionService;
        $this->menuId = Menu::where('route', 'headGroups.index')->value('id');
        $this->currentYear = AccountingYear::current();
    }

    public function index()
    {
        return view('headGroups.index');
    }

    public function create()
    {
        $organisations = Organisation::where('status', 'active')->get();
        return view('headGroups.create', [
            'organisations' => $organisations
        ]);
    }

    public function store(StoreHeadGroupRequest $request): RedirectResponse
    {
        try {
            $validated = $request->validated();

            $headGroup = HeadGroup::create([
                'organisation_id' => $validated['organisation_id'],
                'name'            => trim($validated['name'])
            ]);

            return redirect()
                ->route('head-groups.index')
                ->with('success', 'Head Group "' . $headGroup->name . '" created successfully.');
        } catch (Exception $e) {
            // Return a friendly error message
            return redirect()
                ->back()
                ->withInput()
                ->with('error', 'Something went wrong while creating the Head Group. Please try again.');
        }
    }

    public function edit($encodedId)
    {
        try {
            $id = base64_decode($encodedId);
            if (!is_numeric((int)$id)) {
                abort(404, 'Invalid ID');
            }
            $headGroup = HeadGroup::findOrFail($id);
            $organisations = Organisation::where('status', 'active')->get();
            return view('headGroups.edit', compact(['headGroup', 'organisations']));
        } catch (\Exception $e) {
            abort(404, 'Invalid or corrupted ID');
        }
    }

    public function update(Request $request): RedirectResponse
    {
        try {
            // 🔹 Validate incoming data
            $validator = Validator::make($request->all(), [
                'id' => 'required|exists:head_groups,id',
                'name' => 'required|string|max:100',
            ]);

            if ($validator->fails()) {
                return redirect()
                    ->back()
                    ->withErrors($validator)
                    ->withInput();
            }

            // 🔹 Find the existing record
            $headGroup = HeadGroup::findOrFail($request->id);

            // 🔹 Update its values
            $headGroup->update([
                'name' => trim($request->name),
            ]);

            // 🔹 Redirect back with success
            return redirect()
                ->route('head-groups.index')
                ->with('success', 'Head Group "' . $headGroup->name . '" updated successfully.');
        } catch (Exception $e) {
            // 🔹 Handle unexpected errors
            return redirect()
                ->back()
                ->withInput()
                ->with('error', 'Something went wrong while updating the Head Group. Please try again.');
        }
    }

    public function delete(Request $request)
    {
        $request->validate([
            'id' => 'required|exists:head_groups,id'
        ]);

        try {
            $headGroup = HeadGroup::find($request->id);


            if (!empty($headGroup) && $headGroup->delete()) {
                return response()->json([
                    'status' => true,
                    'message' => 'Deleted successfully',
                    'data' => $headGroup,
                ]);
            } else {
                return response()->json([
                    'status' => false,
                    'message' => 'Failed to delete. Please try again.',
                ], 500);
            }
        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'An error occurred: ' . $e->getMessage(),
            ], 500);
        }
    }
    public function changeStatus(Request $request)
    {
        $request->validate([
            'id' => 'required|exists:head_groups,id',
            'new_status' => 'required|in:active,inactive'
        ]);

        try {
        
            $headGroup = HeadGroup::findOrFail($request->id);

         
            $headGroup->status = $request->new_status;
            $headGroup->save();

            return response()->json([
                'status' => true,
                'message' => 'Status changed successfully',
                'data' => $headGroup,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'An error occurred: ' . $e->getMessage(),
            ], 500);
        }
    }
    public function details(Request $request)
    {
        try {
            // if (!$this->permissionService->hasPermission($this->menuId, 'r')) {
            //     abort(403, 'You do not have read access to Income Heads.');
            // }

            $search = $request->input('search');


            $page   = max(1, (int) $request->input('page', 1));
            $size   = max(1, (int) $request->input('size', 10));

            $sortField = $request->input('sorters.0.field', 'id');
            $sortOrder = $request->input('sorters.0.dir', 'desc');


            $result = HeadGroup::query()
                ->leftJoin('organisations', 'head_groups.organisation_id', '=', 'organisations.organisation_id')
                ->when(
                    $search,
                    fn($q) =>
                    $q->where('head_groups.name', 'like', "%$search%")
                )
                ->orderBy($sortField, $sortOrder)
                ->paginate($size, ['head_groups.*', 'organisations.organisation_name as organisation_name'], '', $page);


            return response()->json([
                'data'         => $result->items(),
                'current_page' => $result->currentPage(),
                'last_page'    => $result->lastPage(),
                'per_page'     => $result->perPage(),
                'total'        => $result->total(),
            ]);
        } catch (\Exception $e) {
            // Return the actual error message for debugging
            return response()->json([
                'error' => $e->getMessage(),
                'line'  => $e->getLine(),
                'file'  => $e->getFile()
            ], 500);
        }
    }
}
