<?php

namespace App\Http\Controllers;

use App\Models\Menu;
use App\Models\Asset;
use App\Models\Donor;
use App\Models\Income;
use App\Models\Purpose;
use App\Models\Liability;
use App\Models\IncomeHead;
use App\Models\AccountType;
use App\Models\AssetRecord;
use App\Models\DonorRecord;
use App\Models\PaymentType;
use App\Traits\CommonTrait;
use App\Models\Organisation;
use Illuminate\Http\Request;
use App\Models\AccountingYear;
use App\Models\IncomeHeadGroup;
use App\Models\LiabilityRecord;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Support\Facades\DB;
use App\Services\PermissionService;
use \Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;


class IncomeController extends Controller
{
    protected $permissionService;
    protected $menuId;
    protected $currentYear;
    use CommonTrait;
    public function __construct(PermissionService $permissionService)
    {
        $this->permissionService = $permissionService;
        $this->menuId = Menu::where('route', 'income.index')->value('id');
        $this->currentYear = AccountingYear::current();
    }

    public function index()
    {
        if (!$this->permissionService->hasPermission($this->menuId, 'r')) {
            abort(403, 'You do not have read access to Income Records.');
        }
        $organisations = Organisation::where('status', 'active')->get();
        return view('auth.transactions.income.index', compact('organisations'));
    }

    public function getIncomeDetails(Request $request)
    {
        if (!$this->permissionService->hasPermission($this->menuId, 'r')) {
            abort(403, 'You do not have read access to Income Records.');
        }

        $search = $request->input('search');
        $organisation_id = $request->input('organisation_id');
        $page   = max(1, (int) $request->input('page', 1));
        $size   = max(1, (int) $request->input('size', 10));

        $sortField = $request->input('sorters.0.field', 'income_id');
        $sortOrder = $request->input('sorters.0.dir', 'desc');

        $result = Income::with('organisation', 'accountType', 'incomeHead', 'paymentType', 'purpose')
            ->where('financial_year_id', $this->currentYear->accounting_year_id)
            ->when($organisation_id, fn($q) => $q->where('organisation_id', $organisation_id))
            ->when(
                $search,
                fn($q) =>
                $q->where(function ($query) use ($search) {
                    $query->where('transaction_reference', 'like', "%{$search}%")
                        ->orWhere('transaction_amount', 'like', "%{$search}%")
                        ->orWhere('transaction_date', 'like', "%{$search}%")
                        ->orWhere('transaction_narration', 'like', "%{$search}%")
                        ->orWhereHas(
                            'organisation',
                            fn($q) =>
                            $q->where('organisation_name', 'like', "%{$search}%")
                        )
                        ->orWhereHas(
                            'accountType',
                            fn($q) =>
                            $q->where('account_type_name', 'like', "%{$search}%")
                        )
                        ->orWhereHas(
                            'incomeHead',
                            fn($q) =>
                            $q->where('income_head_name', 'like', "%{$search}%")
                        );
                })
            )
            ->orderBy($sortField, $sortOrder)
            ->paginate($size, ['*'], '', $page);


        return response()->json([
            'data'         => $result->items(),
            'current_page' => $result->currentPage(),
            'last_page'    => $result->lastPage(),
            'per_page'     => $result->perPage(),
            'total'        => $result->total(),
        ]);
    }

    public function create()
    {
        if (!$this->permissionService->hasPermission($this->menuId, 'w')) {
            abort(403, 'You do not have write access to Income Records.');
        }
        $currentYear = $this->currentYear->accounting_year_financial;

        // return $currentYearId;
        $purposes = Purpose::where('status', 'active')->where('purpose_for', 'income')
            ->get(['id', 'name']);
        $paymentTypes = PaymentType::where('status', 'active')->get();
        $organisations = Organisation::where('status', 'active')->get();
        $accountTypes =  $this->getAccountType();
        $incomeHeads = IncomeHead::where(['status'=>'active','income_ac_type_flg'=>'general_income_heads'])->get();
        $incomeHeadGroups = IncomeHeadGroup::where('status', 'active')->get();
        return view('auth.transactions.income.create', compact(
            'organisations',
            'accountTypes',
            'incomeHeads',
            'currentYear',
            'purposes',
            'paymentTypes',
            'incomeHeadGroups'
        ));
    }



    public function store(Request $request)
    {
        DB::beginTransaction();

        try {
            // --------------------------
            // 1. Handle File Uploads
            // --------------------------
            $filePaths = [];

            if ($request->hasFile('income_files')) {
                foreach ($request->file('income_files') as $file) {
                    $filename = time() . '_' . $file->getClientOriginalName();
                    $path = $file->storeAs('income_files', $filename, 'public');
                    $filePaths[] = $path;
                }
            }

            $filePathString = !empty($filePaths) ? implode(',', $filePaths) : null;

            // --------------------------
            // 2. Split Income Head ID
            // --------------------------
            [$incomeHeadId, $incomeGroupHeadId] = array_map('trim', explode('#', $request->income_head_id));

            // --------------------------
            // 3. Create Income Record
            // --------------------------
            $income = Income::create([
                'organisation_id'       => $request->organisation_id,
                'financial_year_id'     => $this->currentYear->accounting_year_id,
                'account_type_id'       => $request->account_type_id ?? null,
                'income_head'           => $incomeHeadId,
                'income_head_group_id'  => $incomeGroupHeadId,
                'user_id'               => Auth::id(),
                'transaction_amount'    => $request->transaction_amount,
                'payment_type_id'       => $request->payment_type_id,
                'transaction_date'      => $request->transaction_date ?? null,
                'transaction_reference' => $request->transaction_reference ?? null,
                'transaction_narration' => $request->transaction_narration ?? null,
                'money_receipt_no'      => $request->money_receipt_no ?? null,
                'file_path'             => $filePathString,
                'income_type'           => 'income',
            ]);

            // --------------------------
            // 4. Handle Liability Logic
            // --------------------------
            if ($incomeGroupHeadId === '3') {
                // Create Liability entry
                $liability = Liability::create([
                    'organisation_id'   => $request->organisation_id,
                    'account_type_id'   => $request->account_type_id,
                    'income_head_id'    => $incomeHeadId,
                    'liability_amount'  => $request->transaction_amount,
                    'liability_date'    => $request->transaction_date,
                    'financial_year_id' => $this->currentYear->accounting_year_id,
                ]);

                // Create Liability Record
                LiabilityRecord::create([
                    'journal_id'     => $income->income_id,
                    'liability_id'   => $incomeHeadId,
                    'amount'         => $request->transaction_amount,
                    'income_expense' => 'income',
                    'dr_cr'          => 'cr',
                ]);

                // Update Income with liability details
                $income->update([
                    'income_type'  => 'liability',
                ]);
            }

            // --------------------------
            // 5. Commit Transaction
            // --------------------------
            DB::commit();
            return response()->json([
                'success' => true,
                'message' => 'Income entry recorded successfully.',
                'data'    => $income,
            ], 201);

        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'success' => false,
                'message' => 'Failed to record income entry.',
                'error'   => $e->getMessage(),
            ], 500);
        }
    }

    public function eachDetail($id)
    {
        if (!$this->permissionService->hasPermission($this->menuId, 'r')) {
            abort(403, 'You do not have read access to Income Records.');
        }
        $id = base64_decode($id);
        // return $id;

        $donors = Donor::select('donor_id', 'donor_name', 'donor_type')->where('donor_status', 'active')->get();
        $organisations = Organisation::where('status', 'active')->get();
        $accountTypes = AccountType::where('status', 'active')->get();
        $incomeHeads = IncomeHead::where('status', 'active')->get();
        $income = Income::with('paymentType', 'purpose')->where('income_id', $id)->first();
        // return $income;
        $accounting_years = AccountingYear::where('status', 'active')->get();
        return view('auth.transactions.income.detail', compact('donors', 'organisations', 'accountTypes', 'income', 'accounting_years', 'incomeHeads'));
    }
    public function show(string $id)
    {
        //
    }
    public function edit(string $id)
    {
        if (!$this->permissionService->hasPermission($this->menuId, 'x')) {
            abort(403, 'You do not have write access to Income Records.');
        }
        $id = base64_decode($id);
        $assets = Asset::select('asset_id', 'asset_name')->get();
        $liabilities = Liability::select('liability_id', 'liability_name')->where('liability_status', 'active')->get();
        $donors = Donor::select('donor_id', 'donor_name', 'donor_type')->where('donor_status', 'active')->get();
        $income = Income::where('income_id', $id)->first();
        $existingFiles = [];
        if ($income && $income->file_path) {
            $existingFiles = explode(',', $income->file_path);
        }
        $currentYear = $this->currentYear->accounting_year_financial;
        $purposes = Purpose::where('status', 'active')->where('purpose_for', 'income')
            ->get(['id', 'name']);
        $paymentTypes = PaymentType::where('status', 'active')->get();
        // return $currentYearId;
        $incomeHeadGroups = IncomeHeadGroup::where('status', 'active')->get();
        $organisations = Organisation::where('status', 'active')->get();
        $accountTypes = AccountType::where('status', 'active')->get();
        $incomeHeads = IncomeHead::where('status', 'active')->get();
        return view('auth.transactions.income.edit', compact('incomeHeadGroups', 'existingFiles', 'donors', 'assets', 'liabilities', 'organisations', 'accountTypes', 'incomeHeads', 'currentYear', 'income', 'purposes', 'paymentTypes'));
    }
    public function update(Request $request, string $id)
    {
        //  return $request;

        DB::beginTransaction();

        try {
            $income = Income::findOrFail($id);

            AssetRecord::where('journal_id', $income->income_id)->delete();
            LiabilityRecord::where('journal_id', $income->income_id)->delete();
            DonorRecord::where('journal_id', $income->income_id)->delete();

            $filePaths = [];

            if ($request->hasFile('income_files')) {
                if (!empty($income->file_path)) {
                    $existingFiles = explode(',', $income->file_path);
                    foreach ($existingFiles as $oldFile) {
                        if (Storage::disk('public')->exists($oldFile)) {
                            Storage::disk('public')->delete($oldFile);
                        }
                    }
                }

                foreach ($request->file('income_files') as $file) {
                    $filename = time() . '_' . $file->getClientOriginalName();
                    $path = $file->storeAs('income_files', $filename, 'public');
                    $filePaths[] = $path;
                }
            }

            $filePathString = !empty($filePaths) ? implode(',', $filePaths) : null;
            // $income->file_path = $filePathString;
            // $income->save();

            $income->update([
                'organisation_id'       => $request->organisation_id,
                'financial_year_id'     => $this->currentYear->accounting_year_id,
                'account_type_id'       => $request->account_type_id ?? null,
                'income_head'           => $request->income_head_id,
                'transaction_amount'    => $request->transaction_amount,
                'payment_type_id'       => $request->payment_type_id,
                'transaction_date'      => $request->transaction_date ?? null,
                'transaction_reference' => $request->transaction_reference ?? null,
                'purpose_id'            => $request->transaction_purpose_id,
                'transaction_narration' => $request->transaction_narration ?? null,
                'money_receipt_no'      => $request->money_receipt_no ?? null,
                'file_path'             => $filePathString,
                'user_id'               => auth()->id(),
                // reset these (will be set again below)
                'income_type'           => 'income',
                'asset_id'              => null,
                'liability_id'          => null,
                'paid_by'               => null,
                'donor_id'              => null,
            ]);

            if ($request->asset_id && $request->donor_id) {
                AssetRecord::create([
                    'journal_id'    => $id,
                    'asset_id'      => $request->asset_id,
                    'donor_id'      => $request->donor_id,
                    'amount'        => $request->transaction_amount,
                    'income_expense' => 'income',
                    'dr_cr'         => 'cr',
                ]);

                DonorRecord::create([
                    'journal_id'    => $id,
                    'donor_id'      => $request->donor_id,
                    'asset_id'      => $request->asset_id,
                    'paid_amount'   => $request->transaction_amount,
                    'income_expense' => 'income',
                    'dr_cr'         => 'dr',
                ]);

                $income->update([
                    'income_type' => 'asset',
                    'asset_id'    => $request->asset_id,
                    'paid_by'     => 'donor',
                    'donor_id'    => $request->donor_id,
                ]);
            } elseif ($request->liability_id && $request->donor_id) {
                LiabilityRecord::create([
                    'journal_id'    => $id,
                    'liability_id'  => $request->liability_id,
                    'donor_id'      => $request->donor_id,
                    'amount'        => $request->transaction_amount,
                    'income_expense' => 'income',
                    'dr_cr'         => 'cr',
                ]);

                DonorRecord::create([
                    'journal_id'    => $id,
                    'donor_id'      => $request->donor_id,
                    'liability_id'  => $request->liability_id,
                    'paid_amount'   => $request->transaction_amount,
                    'income_expense' => 'income',
                    'dr_cr'         => 'dr',
                ]);

                $income->update([
                    'income_type' => 'liability',
                    'liability_id' => $request->liability_id,
                    'paid_by'     => 'donor',
                    'donor_id'    => $request->donor_id,
                ]);
            } elseif (!$request->asset_id && !$request->liability_id && $request->donor_id) {
                DonorRecord::create([
                    'journal_id'    => $id,
                    'donor_id'      => $request->donor_id,
                    'paid_amount'   => $request->transaction_amount,
                    'income_expense' => 'income',
                    'dr_cr'         => 'dr',
                ]);

                $income->update([
                    'income_type' => 'income',
                    'paid_by'     => 'donor',
                    'donor_id'    => $request->donor_id,
                ]);
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Income updated successfully!',
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Failed to store income',
                'error'   => $e->getMessage()
            ], 500);
        }
    }
    public function destroy(string $id)
    {
        if (!$this->permissionService->hasPermission($this->menuId, 'x')) {
            abort(403, 'You do not have write access to Income Records.');
        }
        // return $id;
        $transaction = Income::findOrFail($id);
        DB::table('donor_records')
            ->where('journal_id', $transaction->income_id)
            ->delete();
        $transaction->delete();

        return response()->json([
            'status'  => true,
            'message' => 'Income deleted successfully.'
        ]);
    }
    public function print($id)
    {
        $income = Income::with(['paymentType'])->findOrFail($id);

        $organisations = Organisation::all();
        $accountTypes = AccountType::all();
        $incomeHeads  = IncomeHead::all();
        $assets       = Asset::all();
        $liabilities  = Liability::all();
        $donors       = Donor::all();
        $accounting_years = AccountingYear::all();

        $pdf = Pdf::loadView('auth.transactions.income.print', compact(
            'income',
            'organisations',
            'accountTypes',
            'incomeHeads',
            'assets',
            'liabilities',
            'donors',
            'accounting_years'
        ));

        return $pdf->stream('income-transaction-' . $income->income_id . '.pdf');
    }
}
