<?php

namespace App\Http\Controllers;

use App\Models\AccountingYear;
use App\Models\AccountType;
use App\Models\IncomeHead;
use App\Models\IncomeHeadGroup;
use App\Models\Menu;
use App\Models\Organisation;
use App\Services\PermissionService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use App\Traits\CommonTrait;

class IncomeHeadController extends Controller
{
    protected $permissionService;
    protected $menuId;
    protected $currentYear;
    use CommonTrait;

    public function __construct(PermissionService $permissionService)
    {
        $this->permissionService = $permissionService;
        $this->menuId = Menu::where('route', 'incomehead.index')->value('id');
        $this->currentYear = AccountingYear::current();
    }


    public function index()
    {
        if (!$this->permissionService->hasPermission($this->menuId, 'r')) {
            abort(403, 'You do not have read access to Income Head Records.');
        }
        $organisations = Organisation::where('status', 'active')->get();
        return view('auth.incomeHead.index', compact('organisations'));
    }
    public function getIncomeHeadDetails(Request $request)
    {
        try {
            if (!$this->permissionService->hasPermission($this->menuId, 'r')) {
                abort(403, 'You do not have read access to Income Heads.');
            }

            $search = $request->input('search');
            $organisation_id = $request->input('organisation_id');

            $page   = max(1, (int) $request->input('page', 1));
            $size   = max(1, (int) $request->input('size', 10));

            $sortField = $request->input('sorters.0.field', 'income_head_id');
            $sortOrder = $request->input('sorters.0.dir', 'desc');


            $query = IncomeHead::query()
                ->leftJoin('account_types', 'account_types.account_type_id', '=', 'income_heads.account_type_id')
                ->leftJoin('income_head_groups', 'income_head_groups.income_head_group_id', '=', 'income_heads.income_head_group_id')
                ->select(
                    'income_heads.*',
                    'account_types.account_type_Name as account_type_name',
                    'income_head_groups.income_head_group_Name as group_name'
                )
                ->when($organisation_id, fn($q) => $q->where('income_heads.organisation_id', $organisation_id))
                ->when(
                    $search,
                    fn($q) =>
                    $q->where('income_heads.income_head_name', 'like', "%$search%")
                        ->orWhere('income_head_groups.income_head_group_Name', 'like', "%$search%")
                )
                ->orderBy($sortField, $sortOrder);

            $result = $query->paginate($size, ['*'], '', $page);

            return response()->json([
                'data'         => $result->items(),
                'current_page' => $result->currentPage(),
                'last_page'    => $result->lastPage(),
                'per_page'     => $result->perPage(),
                'total'        => $result->total(),
            ]);
        } catch (\Exception $e) {
            // Return the actual error message for debugging
            return response()->json([
                'error' => $e->getMessage(),
                'line'  => $e->getLine(),
                'file'  => $e->getFile()
            ], 500);
        }
    }

    public function create()
    {

        $organisations = Organisation::where('status', 'active')->get();
        $accountTypes = $this->getAccountType('general');
        $incomeHeadGroups = IncomeHeadGroup::where('status', 'active')->get();

        return view('auth.incomeHead.create', compact('organisations', 'accountTypes', 'incomeHeadGroups'));
    }

    public function store(Request $request)
    {
        // dd($request);
        try {
            $validator = Validator::make($request->all(), [
                'income_head_name' => 'required|string|max:255|unique:income_heads,income_head_name',
                'account_type_id' => 'nullable|integer',
                'organisation_id' => 'required|integer',
                'income_head_group_id' => 'nullable|integer',
                'head_type' => 'nullable',
                'income_head_group_opening_balance' => 'nullable|numeric',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'errors' => $validator->errors()
                ], 422);
            }

            $id = DB::table('income_heads')->insertGetId([
                'income_head_name' => $request->income_head_name,
                'organisation_id' => $request->organisation_id,
                'account_type_id' => $request->account_type_id ?? null,
                'income_head_group_id' => $request->income_head_group_id ?? null,
                'head_type' => $request->head_type ?? null,
                'income_head_group_opening_balance' => $request->income_head_group_opening_balance ?? 0,
                'status' => $request->status ?? 'active',
                'created_at' => now(),
                'updated_at' => now(),
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Income head added successfully.',
                'new_id' => $id,
                'name' => $request->income_head_name
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Something went wrong!',
                'error' => $e->getMessage()
            ], 500);
        }
    }




    public function edit(string $id)
    {
        //
    }


    public function update(Request $request, string $id)
    {
        //
    }


    public function destroy(string $id)
    {
        //
    }
}
