<?php

namespace App\Http\Controllers;

use App\Models\AccountingYear;
use App\Models\AccountType;
use App\Models\Liability;
use App\Models\Menu;
use App\Models\Organisation;
use App\Services\PermissionService;
use COM;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class LiabilityController extends Controller
{
    protected $permissionService;
    protected $menuId;
    protected $currentYear;

    public function __construct(PermissionService $permissionService)
    {
        $this->permissionService = $permissionService;
        $this->menuId = Menu::where('route', 'liability.index')->value('id');
        $this->currentYear = AccountingYear::current();
    }
    public function index()
    {
        if (!$this->permissionService->hasPermission($this->menuId, 'r')) {
            abort(403, 'You do not have read access to Liability Records.');
        }
        $organisations = Organisation::where('status', 'active')->get();

        return view('auth.liabilities.index', compact('organisations'));
    }

    public function getLiabilityDetails(Request $request)
    {

        $search = $request->input('search');
        $organisation_id = $request->input('organisation_id');

        $page   = max(1, (int) $request->input('page', 1));
        $size   = max(1, (int) $request->input('size', 10));

        $sortField = $request->input('sorters.0.field', 'liability_id');
        $sortOrder = $request->input('sorters.0.dir', 'desc');

        $result = Liability::with('organisation', 'accountType')
            ->where('financial_year_id', $this->currentYear->accounting_year_id)
            ->when($organisation_id, fn($q) => $q->where('organisation_id', $organisation_id))
            ->when(
                $search,
                fn($q) =>
                $q->where(function ($query) use ($search) {
                    $query->where('liability_name', 'like', "%{$search}%")
                        ->orWhere('liability_useful_life', 'like', "%{$search}%")
                        ->orWhereHas(
                            'organisation',
                            fn($q) =>
                            $q->where('organisation_name', 'like', "%{$search}%")
                        )
                        ->orWhereHas(
                            'accountType',
                            fn($q) =>
                            $q->where('account_type_name', 'like', "%{$search}%")
                        );
                })
            )
            ->orderBy($sortField, $sortOrder)
            ->paginate($size, ['*'], '', $page);


        return response()->json([
            'data'         => $result->items(),
            'current_page' => $result->currentPage(),
            'last_page'    => $result->lastPage(),
            'per_page'     => $result->perPage(),
            'total'        => $result->total(),
        ]);
    }
    public function create()
    {
        $organisations = Organisation::where('status', 'active')->get();
        $accountTypes = AccountType::where('status', 'active')->get();
        return view('auth.liabilities.create', compact('organisations', 'accountTypes'));
    }
    public function store(Request $request)
    {
        $rules = [
            'organisation_id' => 'required|integer',
            'account_type_id' => 'nullable|integer',
            'liability_name' => 'required|string|max:255',
            'liability_type' => 'nullable|string|max:255',
            'purchase_date' => 'required|date',
            'liability_opening_balance' => 'nullable|numeric',
            'liability_amount' => 'required|numeric',
            'liability_due_date' => 'nullable|date',
        ];

        // Validate the request
        $validator = Validator::make($request->all(), $rules);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors(),
            ], 422);
        }

        try {
            $liability = new Liability();

            $liability->organisation_id = $request->input('organisation_id');
            $liability->account_type_id = $request->input('account_type_id') ?? null;
            $liability->financial_year_id = $this->currentYear->accounting_year_id;
            $liability->liability_name = $request->input('liability_name');
            $liability->liability_type = $request->input('liability_type') ?? null;
            $liability->liability_date = $request->input('purchase_date');
            $liability->liability_opening_balance = $request->input('liability_opening_balance') ?? 0;
            $liability->liability_amount = $request->input('liability_amount') ?? 0;
            $liability->liability_due_date = $request->input('liability_due_date') ?? null;

            $liability->save();

            return response()->json([
                'success' => true,
                'message' => 'Liability added successfully.',
                'new_id' => $liability->liability_id,
                'name' => $liability->liability_name,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => $e->getMessage(),
            ], 500);
        }
    }

    public function show(string $id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        //
    }
}
