<?php

namespace App\Http\Controllers;

use Exception;
use App\Traits\CommonTrait;
use App\Models\Organisation;
use App\Models\Head;
use App\Models\PaymentType;
use App\Models\AccountingYear;
use App\Models\Bank;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use App\Services\PermissionService;
use Barryvdh\DomPDF\Facade\Pdf;

class JournalTransactionsController extends Controller
{
    protected $permissionService;
    protected $menuId;
    protected $currentYear;
    protected $currentOrgId;
    use  CommonTrait;
    protected $tables; // 👈 Declare it as a class property
    protected $journal_transcation_tbl;
    protected $donor_mapping_tbl;
    protected $vendor_transaction_record;
    protected $assets_transaction_record;
    protected $liability_transaction_record;
    protected $canExecute;

    public function __construct(PermissionService $permissionService)
    {
        $this->tables = $this->getYearPrefixedTables();
        $this->currentOrgId =  Auth::user()->organisation_id;
        $this->currentYear = AccountingYear::current();
        $this->journal_transcation_tbl = $this->tables['journal_transactions'];
        $this->donor_mapping_tbl = $this->tables['donor_transaction_record'];
        $this->vendor_transaction_record = $this->tables['vendor_transaction_record'];
        $this->assets_transaction_record = $this->tables['asset_transaction_record'];
        $this->liability_transaction_record = $this->tables['liability_transaction_record'];
    }

    public function income()
    {
        $organisations = Organisation::where('status', 'active')->get();
        return view('jounaltransactions.income', [
            'organisations' => $organisations
        ]);
    }

    public function expense()
    {
        $organisations = Organisation::where('status', 'active')->get();
        return view('jounaltransactions.expense', [
            'organisations' => $organisations
        ]);
    }

    public function create()
    {
        $organisations = Organisation::where('status', 'active')->orderBy('organisation_name')->get();
        $paymentTypes  = PaymentType::where('status', 'active')->orderBy('name')->get();
        $currentOrganisation = Organisation::current();
        $getHeads      = $this->getHeadsByGroup('General Income', false);
        $banks = $this->getActiveBanks()->toArray();
        $contra_reference_type = [
            "cash_withdraw" => "Cash Withdraw",
            "cash_deposite" => "Cash Deposit",
            "bank_transfer" => "Bank Transfer"
        ];

        return view('jounaltransactions.create', [
            'organisations'                     => $organisations,
            'paymentTypes'                      => $paymentTypes,
            'allHeads'                          => $getHeads,
            'currentOrganisation'               => $currentOrganisation,
            'banks'                             =>$banks,
            'contra_reference_type'             =>$contra_reference_type
        ]);
    }

    public function store(Request $request)
    {   
        // dd($request);

        try {
            $validated = $request->validate([
                'organisation_id'       => 'required|integer|exists:organisations,organisation_id',
                'account_type_id'       => 'nullable|integer|exists:account_types,account_type_ID',
                'head_id'               => 'nullable|integer|exists:heads,id',
                'payment_type_id'       => 'nullable|integer|exists:payment_types,id',
                'transaction_amount'    => 'required|numeric|min:0',
                'transaction_date'      => 'required|date',
                'transaction_reference' => 'nullable|string|max:255',
                'transaction_narration' => 'nullable|string',
                'money_receipt_no'      => 'nullable|string|max:255',
                'transcation_type'      => 'required|in:income,expense,contra',
                'files.*'               => 'nullable|file|mimes:jpg,jpeg,png,pdf|max:10240',
                
                'to_bank_id'            => 'nullable|string|exists:banks,id|different:from_bank_id',
                'from_bank_id'          => 'nullable|string|exists:banks,id|different:to_bank_id',
                'cr_type'               => 'nullable|string|in:cash_deposite,cash_withdraw,bank_transfer',
            ]);

            if (!$this->journal_transcation_tbl) {
                return response()->json([
                    'success' => false,
                    'message' => 'Active financial year not found. Please configure the financial year first.'
                ], 400);
            }


            DB::beginTransaction();

            try {
                // Handle file uploads
                $filePaths = [];
                if ($request->hasFile('files')) {
                    foreach ($request->file('files') as $file) {
                        $filename = time() . '_' . $file->getClientOriginalName();
                        $path = $file->storeAs('journal_transactions', $filename, 'public');
                        $filePaths[] = $path;
                    }
                }
                $filePathString = !empty($filePaths) ? implode(',', $filePaths) : null;

                // Get head group id from head
                $head = Head::find($validated['head_id']);
                $headGroupId = $head ? $head->head_group_id : null;



                // Insert into journal_transactions table
                $journalData = [
                    'organisation_id'       => $validated['organisation_id'],
                    'account_type_id'       => $validated['account_type_id'] ?? null,
                    'head_id'               => $validated['head_id'] ?? null,
                    'head_group_id'         => $headGroupId,
                    'payment_type_id'       => $validated['payment_type_id'] ?? null,
                    'vendor_id'             => $request['vendor_staff_id'] ?? null,
                    'pay_to'                => $request['pay_to'] ?? null,
                    'user_id'               => Auth::id(),
                    'transaction_reference' => $validated['transaction_reference'] ?? null,
                    'transaction_amount'    => $validated['transaction_amount'],
                    'transaction_date'      => $validated['transaction_date'],
                    'transaction_narration' => $validated['transaction_narration'] ?? null,
                    'money_receipt_no'      => $validated['money_receipt_no'] ?? null,
                    'file_path'             => $filePathString,
                    'transcation_type'      => $validated['transcation_type'],

                    'bank_id'               => $validated['to_bank_id'] ?? null,
                    'from_bank_id'          => $validated['from_bank_id'] ?? null,
                    'contra_reference_type' => $validated['cr_type'] ?? null,
                    'created_at'            => now(),
                    'updated_at'            => now(),
                ];



                if (!empty($journalData)) {
                    $insertedId = DB::table($this->journal_transcation_tbl)->insertGetId($journalData);
                }

                if (!empty($validated['vendor_staff_id'])) {
                    $vendorTransactionData = [
                        'journal_id'       => $insertedId ?? null,
                        'asset_head_id'    => $validated['head_id'] ?? null, //for purchase
                        'vendor_id'        => $request['vendor_staff_id'] ?? null,
                        'total_amount'     => $validated['bill_amount'] ?? null,
                        'paid_amount'      => $validated['transaction_amount'] ?? null,
                        'dr_cr'            => 'cr',
                    ];
                    if (!empty($vendorTransactionData)) {
                        DB::table($this->vendor_transaction_record)->insert($vendorTransactionData);
                    }
                }

                //3 means = Asset:purchase
                if (!empty($headGroupId)) {
                    if ($headGroupId == 3) {
                        // Get asset based on head_id to calculate brought forward (opening balance)
                        // Determine dr_cr based on transaction_type
                        // If transaction_type is expense, then dr_cr should be 'cr'
                        $drCr = ($validated['transcation_type'] == 'expense') ? 'cr' : 'dr';

                        // Insert into asset_mapping_transaction table
                        $assetTransactionData = [
                            'vendor_id'   => $request['vendor_staff_id'] ?? null,
                            'journal_id'  => $insertedId ?? null,
                            'amount'      => $validated['transaction_amount'],
                            'dr_cr'       => $drCr,
                            'created_at'  => now(),
                            'updated_at'  => now(),
                        ];

                        if (!empty($assetTransactionData)) {
                            DB::table($this->assets_transaction_record)->insert($assetTransactionData);
                        }
                    }

                    //8 means = Libilioties
                    if ($headGroupId == 8) {
                        $drCr = ($validated['transcation_type'] == 'expense') ? 'cr' : 'dr';
                    }
                }

                DB::commit();
                return response()->json([
                    'success'  => true,
                    'message'  => ucfirst($validated['transcation_type']) . ' transaction saved successfully.',
                    'redirect' => route('transactions.create')
                ], 200);
            } catch (\Exception $e) {
                DB::rollBack();
                throw $e;
            }
        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed' . $e->getMessage(),
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'An error occurred: ' . $e->getMessage()
            ], 500);
        }
    }

    public function list(Request $request)
    {
        try {
            $type          = $request->type ?? 'income';  // 👈 you can pass 'income' or 'expense'
            $organisation_id = $request->organisation_id ?? null;
            $search          = $request->search ?? null;
            $sortField       = $request->sortField ?? 'transaction_date';
            $sortOrder       = $request->sortOrder ?? 'desc';
            $size            = $request->size ?? 10;
            $page            = $request->page ?? 1;

            // 🧠 Use the common query builder
            $query = $this->buildTransactionQuery($type, $organisation_id, $search, $sortField, $sortOrder);
            // 📄 Pagination
            $result = $query->paginate($size, ['*'], 'page', $page);

            return response()->json([
                'status'       => 'success',
                'data'         => $result->items(),
                'current_page' => $result->currentPage(),
                'last_page'    => $result->lastPage(),
                'per_page'     => $result->perPage(),
                'total'        => $result->total(),
            ], 200);
        } catch (\Illuminate\Database\QueryException $e) {
            return response()->json([
                'status'  => 'error',
                'message' => 'Database query error: ' . $e->getMessage(),
            ], 500);
        } catch (\Exception $e) {
            return response()->json([
                'status'  => 'error',
                'message' => 'An error occurred: ' . $e->getMessage(),
            ], 500);
        }
    }

    // 🧱 Common private function for reusable query builder
    private function buildTransactionQuery(string $type, ?int $organisation_id = null, ?string $search = null, string $sortField = 'transaction_date', string $sortOrder = 'desc')
    {
        $table = $this->journal_transcation_tbl;

        if (!$table) {
            throw new \Exception("Transaction table is not defined.");
        }

        $query = DB::table($table . ' as i')
            ->leftJoin('organisations as o', 'i.organisation_id', '=', 'o.organisation_id')
            ->leftJoin('account_types as at', 'i.account_type_id', '=', 'at.account_type_id')
            ->leftJoin('heads as ih', 'i.head_id', '=', 'ih.id')
            ->leftJoin('payment_types as pt', 'i.payment_type_id', '=', 'pt.id')
            ->select(
                'i.*',
                'o.organisation_name',
                'at.account_type_Name',
                'ih.name as head_name',
                'i.transaction_date',
                'i.transaction_narration',
                'i.transaction_amount',
                'i.transcation_type',
                'pt.name as payment_type_name'
            )
            ->where('i.transcation_type', $type)
            ->whereNull('i.deleted_at')
            ->orderBy('i.' . $sortField, $sortOrder);

        // 🧭 Filter by organisation
        if ($organisation_id) {
            $query->where('i.organisation_id', $organisation_id);
        }

        // 🔍 Search filter
        if ($search) {
            $query->where(function ($q) use ($search) {
                $q->where('i.transaction_reference', 'like', "%{$search}%")
                    ->orWhere('i.transaction_amount', 'like', "%{$search}%")
                    ->orWhere('i.transaction_date', 'like', "%{$search}%")
                    ->orWhere('i.transaction_narration', 'like', "%{$search}%")
                    ->orWhere('o.organisation_name', 'like', "%{$search}%")
                    ->orWhere('at.account_type_name', 'like', "%{$search}%")
                    ->orWhere('ih.name', 'like', "%{$search}%");
            });
        }

        return $query;
    }

    public function incomeDelete(Request $request)
    {
        DB::beginTransaction();
        try {
            $validated = $request->validate([
                'id' => 'required|exists:' . $this->journal_transcation_tbl . ',id',
            ]);

            $isJournalDeleted =  DB::table($this->journal_transcation_tbl)
                ->where('id', $validated['id'])
                ->delete();

            if ($isJournalDeleted) {
                DB::table($this->donor_mapping_tbl)
                    ->where('journal_id', $validated['id'])
                    ->delete();
            }

            DB::commit();
            return response()->json([
                'status' => true,
                'message' => 'Deleted successfully'
            ]);
        } catch (Exception $e) {
            DB::rollBack();
            return response()->json([
                'status' => false,
                'message' => 'Deletion failed',
                'error' => $e->getMessage()
            ], 500);
        }
    }
    public function expenseDelete(Request $request)
    {
        DB::beginTransaction();
        try {
            $validated = $request->validate([
                'id' => 'required|exists:' . $this->journal_transcation_tbl . ',id',
            ]);

            $isJournalDeleted =  DB::table($this->journal_transcation_tbl)
                ->where('id', $validated['id'])
                ->delete();

            if ($isJournalDeleted) {
                DB::table($this->vendor_transaction_record)
                    ->where('journal_id', $validated['id'])
                    ->delete();

                DB::table($this->assets_transaction_record)
                    ->where('journal_id', $validated['id'])
                    ->delete();

                DB::table($this->liability_transaction_record)
                    ->where('journal_id', $validated['id'])
                    ->delete();
            }
            DB::commit();
            return response()->json([
                'status' => true,
                'message' => 'Deleted successfully'
            ]);
        } catch (Exception $e) {

            DB::rollBack();
            return response()->json([
                'status' => false,
                'message' => 'Deletion failed',
                'error' => $e->getMessage()
            ], 500);
        }
    }
    public function eachDetail($id)
    {
        $id = base64_decode($id);
        if (!ctype_digit((string) $id)) {
            abort(404);
        }
        $rows = $this->getTransactionDetails($id);
        return view('jounaltransactions.detail', ['data' => $rows, 'accounting_year' => $this->currentYear->accounting_year_financial]);
    }
    public function print($id)
    {
        $rows = $this->getTransactionDetails($id);
        $pdf =  Pdf::loadView('jounaltransactions.print', ['data' => $rows, 'accounting_year' => $this->currentYear->accounting_year_financial]);
        return $pdf->stream($rows->transcation_type . '-transaction-' . $rows->id . '.pdf');
    }
    private function getTransactionDetails($id)
    {
        $rows = DB::table($this->journal_transcation_tbl . ' as j')
            ->leftJoin('donors as d', function ($join) {
                $join->on('d.donor_id', '=', 'j.donor_id');
            })
            ->leftJoin('heads as h', function ($join) {
                $join->on('h.id', '=', 'j.head_id')
                    ->where('h.status', 'active');
            })
            ->leftJoin('payment_types as pt', 'pt.id', '=', 'j.payment_type_id')
            ->leftJoin('account_types as ac', 'ac.account_type_id', '=', 'j.account_type_id')
            ->leftJoin('organisations as org', 'org.organisation_id', '=', 'j.organisation_id')
            ->where('j.id', $id)
            ->select(
                'j.*',
                'd.donor_name',
                'd.donor_type',
                'ac.account_type_Name',
                'pt.name as payment_type_name',
                'h.name as head_name',
                'org.organisation_name'
            )
            ->first();

        return $rows;
    }
}
