<?php

namespace App\Http\Controllers;

use App\Models\Menu;
use App\Models\Organisation;
use App\Models\PermissionCode;
use App\Models\Role;
use App\Models\User;
use App\Models\UserMenuPermission;
use App\Services\PermissionService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rule;

class UserController extends Controller
{
    protected $permissionService;
    protected $menuId;
    protected $currentOrg;

    public function __construct(PermissionService $permissionService)
    {
        $this->permissionService = $permissionService;

        $this->menuId = Menu::where('route', 'users.index')->value('id');
        $this->currentOrg = Organisation::current();
    }

    public function index()
    {
        if (!$this->permissionService->hasPermission($this->menuId, 'r')) {
            abort(403, 'You do not have read access to Users.');
        }

        $organisations = Organisation::where('status', 'active')->get();

        return view('auth.users.index', compact('organisations'));
    }

    public function getUserDetails(Request $request)
    {
        if (!$this->permissionService->hasPermission($this->menuId, 'r')) {
            abort(403, 'You do not have read access to Users.');
        }

        $search          = $request->input('search');
        $organisation_id = $request->input('organisation_id'); 
        $page            = max(1, (int) $request->input('page', 1));
        $size            = max(1, (int) $request->input('size', 10));

        $sortField = $request->input('sorters.0.field', 'id');
        $sortOrder = $request->input('sorters.0.dir', 'desc');

        $result = User::with('role')
            ->when($organisation_id, fn($q) => $q->where('organisation_id', $organisation_id)) 
            ->when($search, fn($q) => $q->where(function ($query) use ($search) {
                $query->where('name', 'like', "%{$search}%")
                    ->orWhere('email', 'like', "%{$search}%");
            }))
            ->orderBy($sortField, $sortOrder)
            ->paginate($size, ['*'], '', $page);


        return response()->json([
            'data'         => $result->items(),
            'current_page' => $result->currentPage(),
            'last_page'    => $result->lastPage(),
            'per_page'     => $result->perPage(),
            'total'        => $result->total(),
        ]);
    }


    public function create()
    {
        if (!$this->permissionService->hasPermission($this->menuId, 'w')) {
            abort(403, 'You do not have write access to Users.');
        }
        $roles = Role::where('status', 'active')->get();
        $menus = Menu::with('children')->where('parent_id', 0)->get();
        $permissionCodes = PermissionCode::all();
        return view('auth.users.create', compact('menus', 'permissionCodes', 'roles'));
    }

    public function store(Request $request)
    {

        // return $this->currentOrg->organisation_id;
        if (!$this->permissionService->hasPermission($this->menuId, 'w')) {
            abort(403, 'You do not have write access to Users.');
        }
        // return $request;
        $validated = $request->validate([
            'fullname'          => 'required|string|max:255',
            // 'username'          => 'required|string|max:255|unique:users,username',
            'role_id' => 'required',
            'email'             => [
                'required',
                'string',
                'email:rfc,dns',
                'max:255',
                'unique:users,email',
            ],
            'password'          => 'required|min:6|confirmed',
            'permission_codes'  => 'nullable|array',
            'permission_codes.*' => 'nullable|array',
        ]);
        if (!$this->currentOrg || !$this->currentOrg->organisation_id) {
            return redirect()->back()
                ->withInput()
                ->withErrors(
                    ['organisation_id' => 'Please select a current organisation before adding user.'],
                    'orgCheck'
                );
        }

        $user = User::create([
            'name' => $validated['fullname'],
            'username' => $validated['fullname'],
            'email'    => $validated['email'],
            'password' => Hash::make($validated['password']),
            'status'   => 'active',
            'role_id' => $validated['role_id'],
            'organisation_id' => $this->currentOrg->organisation_id

        ]);
        return redirect()->route('users.index')->with('success', 'User created successfully.');
    }


    public function show(string $id)
    {
        //
    }


    public function edit(string $id)
    {
        if (!$this->permissionService->hasPermission($this->menuId, 'x')) {
            abort(403, 'You do not have execute access to Users.');
        }

        // return $id;
        $id = base64_decode($id);
        // return $id;
        $user = User::find($id);
        $roles = Role::where('status', 'active')->get();
        $menus = Menu::with('children')->where('parent_id', 0)->get();
        $permissionCodes = PermissionCode::all();
        return view('auth.users.edit', compact('user', 'menus', 'permissionCodes', 'roles'));
    }


    public function update(Request $request, string $id)
    {
        if (!$this->permissionService->hasPermission($this->menuId, 'x')) {
            abort(403, 'You do not have execute access to Users.');
        }

        $user = User::findOrFail($id);

        $validated = $request->validate([
            'fullname'          => 'required|string|max:255',
            'role_id'           => 'required',
            'email'             => [
                'required',
                'string',
                'email:rfc,dns',
                'max:255',
                Rule::unique('users', 'email')->ignore($user->id),
            ],
            'password'          => 'nullable|min:6|confirmed',
            'permission_codes'  => 'nullable|array',
            'permission_codes.*' => 'nullable|array',
        ]);

        $updateData = [
            'name'     => $validated['fullname'],
            'username' => $validated['fullname'],
            'email'    => $validated['email'],
            'role_id'  => $validated['role_id'],
        ];

        if ($request->filled('password') && $request->filled('password_confirmation')) {
            if ($request->password === $request->password_confirmation) {
                $updateData['password'] = Hash::make($request->password);
            } else {
                return back()->withErrors(['password' => 'Password confirmation does not match.']);
            }
        }

        $user->update($updateData);
        return redirect()->route('users.index')->with('success', 'User updated successfully.');
    }


    public function destroy(string $id)
    {
        if (!$this->permissionService->hasPermission($this->menuId, 'x')) {
            abort(403, 'You do not have execute access to Users.');
        }
    }

    public function permissions($id)
    {
        if (!$this->permissionService->hasPermission($this->menuId, 'x')) {
            abort(403, 'You do not have execute access to Users.');
        }
        $id = base64_decode($id);

        $user = User::with('menuPermissions.menu')->find($id);
        //  return $user;
        $menus = Menu::with('children')->where('parent_id', 0)->get();
        $permissionCodes = PermissionCode::all();
        return view('auth.users.permission', compact('menus', 'permissionCodes', 'user'));
    }

    public function permissionsStore(Request $request, $userId)
    {
        if (!$this->permissionService->hasPermission($this->menuId, 'x')) {
            abort(403, 'You do not have execute access to Users.');
        }
        // return $request;
        $user = User::findOrFail($userId);
        $permissionMap = [
            1 => ['r'],
            2 => ['r', 'w'],
            3 => ['r', 'w', 'x'],
        ];

        UserMenuPermission::where('user_id', $userId)->delete();

        if ($request->has('permission_codes')) {
            $insertData = [];
            foreach ($request->permission_codes as $menuId => $codes) {
                foreach ($codes as $codeId) {
                    $insertData[] = [
                        'user_id'     => $userId,
                        'menu_id'     => $menuId,
                        'permissions' => $codeId,
                        'created_at'  => now(),
                        'updated_at'  => now(),
                    ];
                }
            }

            if (!empty($insertData)) {
                UserMenuPermission::insert($insertData);
            }
        }
        return redirect()->route('users.index')->with('success', 'Permissions updated successfully.');
    }

    public function status($id)
    {
        if (!$this->permissionService->hasPermission($this->menuId, 'x')) {
            abort(403, 'You do not have execute access to Users.');
        }

        $user = User::findOrFail($id);

        if ($user->role_id == 1 && $user->status === 'active') {
            $activeSuperAdmins = User::where('role_id', 1)
                ->where('status', 'active')
                ->count();

            if ($activeSuperAdmins <= 1) {
                return response()->json([
                    'success' => false,
                    'message' => 'At least one active super admin must exist.',
                ], 422);
            }
        }

        $user->status = $user->status === 'active' ? 'inactive' : 'active';
        $user->save();

        return response()->json([
            'success' => true,
            'status' => $user->status,
        ]);
    }
}
