<?php

namespace App\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\Rule;

class StoreContraTransactionRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return true;
    }

    /**
     * Prepare the data for validation.
     */
    protected function prepareForValidation(): void
    {
        // Ensure organisation_id is an integer (only if not empty)
        if ($this->has('organisation_id') && $this->organisation_id !== '' && $this->organisation_id !== null) {
            $this->merge([
                'organisation_id' => (int) $this->organisation_id,
            ]);
        }

        // Ensure transaction_amount is numeric (only if not empty)
        if ($this->has('transaction_amount') && $this->transaction_amount !== '' && $this->transaction_amount !== null) {
            $amount = (float) $this->transaction_amount;
            if ($amount > 0) {
                $this->merge([
                    'transaction_amount' => $amount,
                ]);
            }
        }

        // Ensure bank_id is an integer (only if not empty)
        if ($this->has('bank_id') && $this->bank_id !== '' && $this->bank_id !== null && $this->bank_id !== '0') {
            $this->merge([
                'bank_id' => (int) $this->bank_id,
            ]);
        }

        // Ensure from_bank_id is an integer (only if not empty)
        if ($this->has('from_bank_id') && $this->from_bank_id !== '' && $this->from_bank_id !== null && $this->from_bank_id !== '0') {
            $this->merge([
                'from_bank_id' => (int) $this->from_bank_id,
            ]);
        }

    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array<string, \Illuminate\Contracts\Validation\ValidationRule|array<mixed>|string>
     */
    public function rules(): array
    {
        $type = $this->input('type');

        return [
            'organisation_id' => [
                'required',
                'integer',
                'exists:organisations,organisation_id'
            ],

            'type' => [
                'required',
                'string',
                Rule::in(['cash_deposite', 'cash_withdraw', 'bank_transfer'])
            ],

            'transaction_amount' => [
                'required',
                'numeric',
                'min:0.01'
            ],

            'bank_id' => [
                'required',
                'integer',
                'exists:banks,id'
            ],

            'from_bank_id' => [
                Rule::requiredIf($type === 'bank_transfer'),
                'nullable',
                'integer',
                'exists:banks,id'
            ],

            'transaction_date' => [
                'required',
                'date'
            ],

            'transaction_reference' => [
                'nullable',
                'string',
                'max:255'
            ],

            'transaction_narration' => [
                'nullable',
                'string'
            ],
        ];
    }

    /**
     * Get custom messages for validator errors.
     *
     * @return array<string, string>
     */
    public function messages(): array
    {
        return [
            'organisation_id.required' => 'Organisation is required.',
            'organisation_id.integer' => 'Organisation must be an integer ID.',
            'organisation_id.exists' => 'Selected organisation does not exist.',

            'type.required' => 'Transaction type is required.',
            'type.in' => 'Transaction type must be one of: Cash Deposite, Cash Withdraw, or Bank Transfer.',

            'transaction_amount.required' => 'Transaction amount is required.',
            'transaction_amount.numeric' => 'Transaction amount must be a valid number.',
            'transaction_amount.min' => 'Transaction amount must be at least 0.01.',

            'bank_id.required' => 'Bank selection is required.',
            'bank_id.integer' => 'Bank ID must be an integer.',
            'bank_id.exists' => 'Selected bank does not exist.',

            'from_bank_id.required_if' => 'From bank is required for bank transfer transactions.',
            'from_bank_id.integer' => 'From bank ID must be an integer.',
            'from_bank_id.exists' => 'Selected from bank does not exist.',

            'transaction_date.required' => 'Transaction date is required.',
            'transaction_date.date' => 'Transaction date must be a valid date.',

            'transaction_reference.max' => 'Transaction reference cannot exceed 255 characters.',
        ];
    }

    /**
     * Configure the validator instance.
     *
     * @param  \Illuminate\Validation\Validator  $validator
     * @return void
     */
    public function withValidator($validator)
    {
    }
}

