<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

use App\Models\SchoolClass;
use App\Models\SchoolSection;

use App\Models\StudentClassDetail;
use App\Models\ClassFeeConfig;
use Illuminate\Http\JsonResponse;

class ClassSetupController extends Controller
{
    public function index()
    {
        $classes = SchoolClass::with(['sections', 'students'])->get();

        return view('class.index', compact('classes'));
    }

    public function addClass(Request $request)
    {
        $request->validate([
            'class_name' => 'required|unique:classes,class_name'
        ]);

        SchoolClass::create([
            'class_name' => $request->class_name
        ]);

        return back()->with('success', 'Class added successfully.');
    }


    public function updateClass(Request $request, $id)
    {
        $class = SchoolClass::findOrFail($id);

        $request->validate([
            'class_name' => 'required|string|max:255|unique:classes,class_name,' . $id,
        ]);

        $class->update([
            'class_name' => $request->class_name,
        ]);

        return response()->json(['success' => true]);
    }


    // public function deleteClass($id)
    // {
    //     SchoolClass::findOrFail($id)->delete();

    //     return response()->json(['success' => true]);
    // }



    public function deleteClass($id): JsonResponse
    {
        $class = SchoolClass::findOrFail($id);

        /* --------------------------------------------
            Check students assigned to this class
        -------------------------------------------- */
        $hasStudents = StudentClassDetail::where('class_id', $id)
            ->where('status', 1) // active students only
            ->exists();

        if ($hasStudents) {
            return response()->json([
                'success' => false,
                'message' => 'Cannot delete class. Students are already assigned to this class.'
            ], 422);
        }

        /* --------------------------------------------
            Check fee configuration exists
        -------------------------------------------- */
        $hasFeeConfig = ClassFeeConfig::where('class_id', $id)->exists();

        if ($hasFeeConfig) {
            return response()->json([
                'success' => false,
                'message' => 'Cannot delete class. Fee configuration exists for this class.'
            ], 422);
        }

        /* --------------------------------------------
            Safe to delete
        -------------------------------------------- */
        $class->delete();

        return response()->json([
            'success' => true,
            'message' => 'Class deleted successfully.'
        ]);
    }


    

    public function addSection(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'class_id'     => 'required',
            'section_name' => 'required|string|max:255',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'errors' => $validator->errors()
            ], 422);
        }

        // 🔒 Class-wise duplicate check
        $exists = SchoolSection::where('class_id', $request->class_id)
            ->where('section_name', $request->section_name)
            ->exists();

        if ($exists) {
            return response()->json([
                'errors' => [
                    'section_name' => ['This section already exists for the selected class']
                ]
            ], 422);
        }

        $section = SchoolSection::create([
            'class_id'     => $request->class_id,
            'section_name' => $request->section_name,
            'status'       => 1
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Section added successfully',
            'data'    => $section
        ]);
    }




    public function deleteSection($id): JsonResponse
    {
        $section = SchoolSection::findOrFail($id);

        /* --------------------------------------------
            Check students assigned to this section
        -------------------------------------------- */
        $hasStudents = StudentClassDetail::where('section_id', $id) 
            ->exists();

        if ($hasStudents) {
            return response()->json([
                'success' => false,
                'message' => 'Cannot delete section. Students are assigned to this section.'
            ], 422);
        }

        /* --------------------------------------------
            Safe to delete
        -------------------------------------------- */
        $section->delete();

        return response()->json([
            'success' => true,
            'message' => 'Section deleted successfully.'
        ]);
    }

}
