<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

use App\Models\AcademicYear;
use App\Models\Student;
use App\Models\Staff;
use App\Models\StudentMonthlyFee;
use App\Models\ClassFeeConfig;
use App\Models\StudentClassDetail;

class DashboardController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth')->except([
            'showLogin', 'login', 'showRegister', 'doRegister'
        ]);
    }

    public function index()
    {
        /* -------------------------------------------------
            Active Academic Year
        ------------------------------------------------- */
        $academicYear = AcademicYear::where('status', 1)->firstOrFail();

        $financialYearId = $academicYear->id;
        $todayMonth = now()->format('Y-m');

        /* -------------------------------------------------
            Active Students Count
        ------------------------------------------------- */
        $activeStudents = Student::where('status', 1)
            ->whereHas('activeClass', function ($q) use ($financialYearId) {
                $q->where('financial_year_id', $financialYearId)
                  ->where('status', 1);
            })
            ->count();

        /* -------------------------------------------------
            Active Staff Count
        ------------------------------------------------- */
        $activeStaff = Staff::where('status', 1)->count();

        /* -------------------------------------------------
            Total Collected Fee (Till Today)
        ------------------------------------------------- */
        $totalCollectedFee = StudentMonthlyFee::where('financial_year_id', $financialYearId)
            ->where('paid_date', '<=', now())
            ->sum('amount');

        /* -------------------------------------------------
            Total Pending Fee (Till Today)
        ------------------------------------------------- */
        $pendingFee = 0;

        $students = Student::where('status', 1)
            ->whereHas('activeClass', function ($q) use ($financialYearId) {
                $q->where('financial_year_id', $financialYearId)
                  ->where('status', 1);
            })
            ->with('activeClass')
            ->get();

        foreach ($students as $student) {

            $activeClass = $student->activeClass;
            if (!$activeClass) {
                continue;
            }

            $configs = ClassFeeConfig::where([
                'class_id' => $activeClass->class_id,
                'financial_year_id' => $financialYearId,
                'status' => 1,
            ])->get();

            if ($configs->isEmpty()) {
                continue;
            }

            foreach ($configs as $cfg) {

                $applicable = false;

                // recurring fee applies every month
                if ($cfg->fee_type === 'recurring') {
                    $applicable = true;
                }

                // one-time fee applies only in due month
                if (
                    $cfg->fee_type === 'one_time' &&
                    $cfg->due_date &&
                    Carbon::parse($cfg->due_date)->format('Y-m') <= $todayMonth
                ) {
                    $applicable = true;
                }

                if (!$applicable) {
                    continue;
                }

                $paid = StudentMonthlyFee::where([
                    'student_id'        => $student->id,
                    'fee_head_id'       => $cfg->fee_head_id,
                    'financial_year_id' => $financialYearId,
                    'fee_for_month'     => $todayMonth,
                ])->exists();

                if (!$paid) {

                    $lateFee = 0;

                    if ($cfg->fee_type === 'recurring' && $cfg->due_day_of_month) {
                        if (now()->day > $cfg->due_day_of_month) {
                            $lateFee = $cfg->late_fee_amount ?? 0;
                        }
                    }

                    if ($cfg->fee_type === 'one_time' && $cfg->due_date) {
                        if (now()->gt(Carbon::parse($cfg->due_date))) {
                            $lateFee = $cfg->late_fee_amount ?? 0;
                        }
                    }

                    $pendingFee += ($cfg->amount + $lateFee);
                }
            }
        }

        /* -------------------------------------------------
            Return View
        ------------------------------------------------- */
        return view('dashboard.index', [
            'academicYear'       => $academicYear,
            'activeStudents'     => $activeStudents,
            'activeStaff'        => $activeStaff,
            'totalCollectedFee'  => $totalCollectedFee,
            'pendingFee'         => $pendingFee,
        ]);
    }
}
