<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

use App\Models\FeeHead;
use App\Models\ClassFeeConfig;
use App\Models\SchoolClass;
use App\Models\AcademicYear;
use App\Models\StudentMonthlyFee;

class FeeController extends Controller
{
    public function index(Request $request)
    {
        $tab = $request->get('tab', 'fee_head');
        $classId = $request->get('class_id');
        $financialYear = $request->get('financial_year_id');
        if($classId || $financialYear){
            $tab = 'class_fee';
        }
        $active_academic_year = AcademicYear::where('status',1)->first();

        $classes = SchoolClass::where('status', 1)->get();
        $academic_years = AcademicYear::where('status',1)->get();

        $feeHeads = FeeHead::orderBy('id', 'desc')
            ->paginate(25, ['*'], 'fee_head_page')
            ->withQueryString();

        $classFees = ClassFeeConfig::with('feeHead','class','financialYear')
            ->when($classId, fn ($q) => $q->where('class_id', $classId))
            ->when($financialYear, fn ($q) => $q->where('financial_year_id', $financialYear))
            ->orderBy('id', 'desc')
            ->paginate(25, ['*'], 'class_fee_page')
            ->withQueryString();

        return view('fees.index', compact(
            'tab',
            'academic_years',
            'classes',
            'feeHeads',
            'classFees',
            'classId',
            'financialYear',
            'active_academic_year'
        ));
    }

    /* =========================
        FEE HEAD
    ========================= */

    public function storeFeeHead(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
        ]);

        $exists = FeeHead::where('name', $request->name)->exists();
        if ($exists) {
            return redirect()->route('fees.index', [
                'tab' => 'fee_head'
            ])->with('error', 'Fee Head already exists!');
        }

        FeeHead::create([
            'name' => $request->name,
            'description' => $request->description,
            'status' => 1
        ]);

        return redirect()->route('fees.index', [
            'tab' => 'fee_head'
        ])->with('success', 'Fee Head Added!');
    }

    /**
     * 🔥 SINGLE FUNCTION
     * Edit + Status Toggle (AJAX)
     */
    public function updateFeeHead(Request $request, $id)
    {
        $feeHead = FeeHead::findOrFail($id);

        /*
        |--------------------------------------------------------------------------
        | STATUS TOGGLE (AJAX)
        |--------------------------------------------------------------------------
        */
        if ($request->has('status')) {

            DB::beginTransaction();

            try {

                // Update fee head status
                $feeHead->update([
                    'status' => $request->status ? 1 : 0
                ]);

                //  If DEACTIVATING → deactivate all class fee configs
                if ((int) $request->status === 0) {
                    ClassFeeConfig::where('fee_head_id', $feeHead->id)
                        ->update(['status' => 0]);
                }

                DB::commit();

                return response()->json([
                    'success' => true,
                    'message' => $request->status
                        ? 'Fee Head activated successfully'
                        : 'Fee Head deactivated and removed from class fee configs'
                ]);

            } catch (\Throwable $e) {

                DB::rollBack();

                return response()->json([
                    'success' => false,
                    'message' => 'Unable to update fee head status'
                ], 500);
            }
        }


        /*
        |--------------------------------------------------------------------------
        | EDIT (AJAX)
        |--------------------------------------------------------------------------
        */
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'errors' => $validator->errors()
            ], 422);
        }

        $exists = FeeHead::where('name', $request->name)
            ->where('id', '!=', $id)
            ->exists();

        if ($exists) {
            return response()->json([
                'errors' => [
                    'name' => ['Fee Head already exists']
                ]
            ], 422);
        }

        $feeHead->update([
            'name' => $request->name,
            'description' => $request->description
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Fee Head updated successfully'
        ]);
    }

    public function deleteFeeHead($id)
    {
        try {
            $feeHead = FeeHead::findOrFail($id);

            /* -----------------------------------------
            Restriction 1: Used in Class Fee Config
            ----------------------------------------- */
            $usedInClassFee = ClassFeeConfig::where('fee_head_id', $id)->exists();

            if ($usedInClassFee) {
                return response()->json([
                    'success' => false,
                    'message' => 'Fee Head is already used in class fee configuration.'
                ], 422);
            }

            /* -----------------------------------------
            Restriction 2: Already collected from students
            ----------------------------------------- */
            $usedInStudentFee = StudentMonthlyFee::where('fee_head_id', $id)->exists();

            if ($usedInStudentFee) {
                return response()->json([
                    'success' => false,
                    'message' => 'Fee Head already collected from students. Deletion not allowed.'
                ], 422);
            }

            $feeHead->delete();

            return response()->json([
                'success' => true,
                'message' => 'Fee Head deleted successfully'
            ]);

        } catch (\Throwable $e) {

            return response()->json([
                'success' => false,
                'message' => 'Unable to delete fee head'
            ], 500);
        }
    }


    /* =========================
        CLASS FEE CONFIG
    ========================= */

    public function storeClassFee(Request $request)
    {
        $rules = [
            'fee_head_id'        => 'required',
            'class_id'           => 'required',
            'financial_year_id'  => 'required',
            'amount'             => 'required|numeric|min:0',
            'late_fee_amount'    => 'nullable|numeric',
            'fee_type'           => 'required|in:recurring,one_time',
        ];

        if ($request->fee_type === 'recurring') {
            $rules['due_day_of_month'] = 'required|integer|min:1|max:31';
        }

        if ($request->fee_type === 'one_time') {
            $rules['due_date'] = 'required|date';
        }

        $validated = $request->validate($rules);

        DB::beginTransaction();

        try {

            /* ------------------------------------------
            APPLY TO ALL ACTIVE CLASSES
            ------------------------------------------ */
            $classIds = $request->class_id === 'all'
                ? SchoolClass::where('status', 1)->pluck('id')
                : collect([$request->class_id]);

            foreach ($classIds as $classId) {

                // ❌ Prevent duplicate (especially one-time)
                $exists = ClassFeeConfig::where('class_id', $classId)
                    ->where('fee_head_id', $request->fee_head_id)
                    ->where('financial_year_id', $request->financial_year_id)
                    ->where('fee_type', $request->fee_type)
                    ->exists();

                if ($exists) {
                    continue; // silently skip duplicates
                }

                ClassFeeConfig::create([
                    'class_id'          => $classId,
                    'fee_head_id'       => $request->fee_head_id,
                    'financial_year_id' => $request->financial_year_id,
                    'fee_type'          => $request->fee_type,
                    'amount'            => $request->amount,
                    'late_fee_amount'   => $request->late_fee_amount,
                    'due_date'          => $request->fee_type === 'one_time'
                                            ? $request->due_date
                                            : null,
                    'due_day_of_month'  => $request->fee_type === 'recurring'
                                            ? $request->due_day_of_month
                                            : null,
                    'status'            => 1
                ]);
            }

            DB::commit();

            return redirect()
                ->route('fees.index', ['tab' => 'class_fee'])
                ->with('success', 'Class fee configured successfully');

        } catch (\Throwable $e) {

            DB::rollBack();

            return redirect()
                ->route('fees.index', ['tab' => 'class_fee'])
                ->with('error', 'Something went wrong while saving fee');
        }
    }



    /**
     * 🔥 SINGLE FUNCTION
     * Edit + Status Toggle
     */


    public function updateClassFee(Request $request, $id)
    {
        DB::beginTransaction();

        try {

            $classFee = ClassFeeConfig::with('feeHead')->findOrFail($id);

            /* ------------------------------------------
            STATUS TOGGLE (AJAX)
            ------------------------------------------ */
            if ($request->exists('status')) {

                // ❌ Block activation if Fee Head inactive
                if (
                    (int) $request->status === 1 &&
                    (int) $classFee->feeHead->status === 0
                ) {
                    DB::rollBack();

                    return response()->json([
                        'success' => false,
                        'message' => 'To reactivate the fee configuration, please activate the Fee Head first.'
                    ], 422);
                }

                $classFee->update([
                    'status' => (int) $request->status
                ]);

                DB::commit();

                return response()->json([
                    'success' => true,
                    'message' => (int) $request->status === 1
                        ? 'Class fee activated successfully'
                        : 'Class fee deactivated successfully'
                ]);
            }

            /* ------------------------------------------
            VALIDATION
            ------------------------------------------ */
            $rules = [
                'class_id'           => 'required',
                'fee_head_id'        => 'required',
                'financial_year_id'  => 'required',
                'amount'             => 'required|numeric|min:0',
                'late_fee_amount'    => 'nullable|numeric|min:0',
                'fee_type'           => 'required|in:recurring,one_time',
            ];

            if ($request->fee_type === 'recurring') {
                $rules['due_day_of_month'] = 'required|integer|min:1|max:31';
            }

            if ($request->fee_type === 'one_time') {
                $rules['due_date'] = 'required|date';
            }

            $validator = Validator::make($request->all(), $rules);

            if ($validator->fails()) {
                DB::rollBack();

                return response()->json([
                    'errors' => $validator->errors()
                ], 422);
            }

            /* ------------------------------------------
            PREVENT UPDATE ON INACTIVE CLASS
            ------------------------------------------ */
            if ($request->class_id !== 'all') {
                $active = SchoolClass::where('id', $request->class_id)
                    ->where('status', 1)
                    ->exists();

                if (!$active) {
                    DB::rollBack();

                    return response()->json([
                        'errors' => ['class_id' => ['Selected class is inactive']]
                    ], 422);
                }
            }

            /* ------------------------------------------
            🔒 BLOCK UPDATE IF ANY STUDENT PAID (ALL CLASSES)
            ------------------------------------------ */
            if ($request->class_id === 'all') {

                $classFeeIds = ClassFeeConfig::where('fee_head_id', $request->fee_head_id)
                    ->where('financial_year_id', $request->financial_year_id)
                    ->whereHas('class', fn ($q) => $q->where('status', 1))
                    ->pluck('id')
                    ->toArray();

                $alreadyPaid = StudentMonthlyFee::whereIn('fee_head_id', [$request->fee_head_id])
                    ->where('financial_year_id', $request->financial_year_id)
                    ->exists();

                if ($alreadyPaid) {
                    DB::rollBack();

                    return response()->json([
                        'success' => false,
                        'message' => 'Fee update not allowed. Some students have already paid this fee.'
                    ], 422);
                }

                $updated = ClassFeeConfig::whereIn('id', $classFeeIds)->update([
                    'fee_type'         => $request->fee_type,
                    'amount'           => $request->amount,
                    'late_fee_amount'  => $request->late_fee_amount,
                    'due_date'         => $request->fee_type === 'one_time'
                                            ? $request->due_date
                                            : null,
                    'due_day_of_month' => $request->fee_type === 'recurring'
                                            ? $request->due_day_of_month
                                            : null,
                ]);

                DB::commit();

                return response()->json([
                    'success' => true,
                    'message' => "Updated {$updated} active classes"
                ]);
            }

            /* ------------------------------------------
            BLOCK UPDATE IF ANY STUDENT PAID (SINGLE CLASS)
            ------------------------------------------ */
            $alreadyPaid = StudentMonthlyFee::where('fee_head_id', $classFee->fee_head_id)
                ->where('financial_year_id', $classFee->financial_year_id)
                ->exists();

            if ($alreadyPaid) {
                DB::rollBack();

                return response()->json([
                    'success' => false,
                    'message' => 'Fee update not allowed. Students have already paid this fee.'
                ], 422);
            }

            /* ------------------------------------------
            DUPLICATE CHECK (SINGLE CLASS)
            ------------------------------------------ */
            $exists = ClassFeeConfig::where('class_id', $request->class_id)
                ->where('fee_head_id', $request->fee_head_id)
                ->where('financial_year_id', $request->financial_year_id)
                ->where('fee_type', $request->fee_type)
                ->where('id', '!=', $id)
                ->exists();

            if ($exists) {
                DB::rollBack();

                return response()->json([
                    'errors' => ['fee_head_id' => ['Duplicate fee configuration']]
                ], 422);
            }

            /* ------------------------------------------
            UPDATE SINGLE RECORD
            ------------------------------------------ */
            $classFee->update([
                'class_id'          => $request->class_id,
                'fee_head_id'       => $request->fee_head_id,
                'financial_year_id' => $request->financial_year_id,
                'fee_type'          => $request->fee_type,
                'amount'            => $request->amount,
                'late_fee_amount'   => $request->late_fee_amount,
                'due_date'          => $request->fee_type === 'one_time'
                                        ? $request->due_date
                                        : null,
                'due_day_of_month'  => $request->fee_type === 'recurring'
                                        ? $request->due_day_of_month
                                        : null,
            ]);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Class fee updated successfully'
            ]);

        } catch (\Throwable $e) {

            DB::rollBack();

            return response()->json([
                'success' => false,
                'message' => 'Unexpected error occurred'
            ], 500);
        }
    }


    public function deleteClassFee($id)
    {
        try {
            $fee = ClassFeeConfig::findOrFail($id);

            $alreadyUsed = StudentMonthlyFee::where('fee_head_id', $fee->fee_head_id)
                ->where('financial_year_id', $fee->financial_year_id)
                ->exists();

            if ($alreadyUsed) {
                return response()->json([
                    'success' => false,
                    'message' => 'Fee already collected for students. Deletion not allowed.'
                ], 422);
            }

            $fee->delete();

            return response()->json([
                'success' => true,
                'message' => 'Class fee deleted successfully'
            ]);
        } catch (\Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Unable to delete class fee'
            ], 500);
        }
    }

}
