<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Staff;
use App\Models\StaffSalary;
use App\Models\AcademicYear;
use App\Models\StaffBankDetails;
use App\Models\StaffAcademicYear;
use App\Models\StaffMonthlySalary;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class StaffController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    /* =========================================================
     * INDEX – show only staff of ACTIVE academic year
     * ========================================================= */
    

    public function index(Request $request)
    {
        /* ---------------- TAB HANDLING ---------------- */
        $tab = $request->get('tab', 'staff');
        session(['tab' => $tab]);

        /* ---------------- ACADEMIC YEAR ---------------- */
        $academic_years = AcademicYear::orderBy('year_from')->where('status',1)->get();

        $activeAcademicYearId = $request->financial_year_id
            ?? AcademicYear::where('status', 1)->value('id');

        /* =========================================================
            STAFF LIST TAB
        ========================================================= */
        $staffs = collect();

        //if ($tab === 'staff') {

            $staffs = Staff::with('activeYear')
                ->where('status', 1)

                // ✅ ONLY staff active in selected academic year
                ->whereHas('academicYears', function ($q) use ($activeAcademicYearId) {
                    $q->where('financial_year_id', $activeAcademicYearId)
                      ->where('status', 1);
                })

                // 🔍 Name filter
                ->when($request->filled('name'), fn ($q) =>
                    $q->where('name', 'like', '%' . $request->name . '%')
                )

                // 🔍 Category filter
                ->when($request->filled('category'), fn ($q) =>
                    $q->where('category', $request->category)
                )

                ->orderBy('name')
                ->paginate(20)
                ->withQueryString();
        //}

        /* =========================================================
            STAFF SALARY TAB
        ========================================================= */
        $salaryStaffs = collect();

        if ($tab == 'staff_salary' && $request->filled('month')) {

            session(['tab' => 'staff_salary']);
            //echo 'jhsgdhjgdh';die;
            $month = $request->month; // YYYY-MM

            $salaryStaffs = Staff::where('status', 1)

                // ✅ Only staff active in this academic year
                ->whereHas('academicYears', function ($q) use ($activeAcademicYearId) {
                    $q->where('financial_year_id', $activeAcademicYearId)
                      ->where('status', 1);
                })

                ->with([
                    // ✅ Monthly salary ONLY for this staff + month + year
                    'monthlySalaries' => function ($q) use ($month, $activeAcademicYearId) {
                        $q->where('pay_for_month', $month)
                          ->where('financial_year_id', $activeAcademicYearId);
                    }
                ])

                ->orderBy('name')
                ->get()

                ->map(function ($staff) use ($month, $activeAcademicYearId) {

                    // 🔹 MONTHLY SALARY (PAID)
                    $monthlySalary = $staff->monthlySalaries->first();

                    if ($monthlySalary) {
                        return [
                            'staff'  => $staff,
                            'salary' => $monthlySalary,
                            'status' => 'paid',
                            'source' => 'monthly',
                        ];
                    }

                    // 🔹 BASE SALARY (PENDING)
                    $baseSalary = StaffSalary::where('staff_id', $staff->id)
                        ->where('financial_year_id', $activeAcademicYearId)
                        ->where('status', 1)
                        ->first();

                    if ($baseSalary) {
                        return [
                            'staff'  => $staff,
                            'salary' => $baseSalary,
                            'status' => 'pending',
                            'source' => 'base',
                        ];
                    }

                    // 🔹 NOT CONFIGURED
                    return [
                        'staff'  => $staff,
                        'salary' => null,
                        'status' => 'not_configured',
                        'source' => 'none',
                    ];
                });
        }

        /* ---------------- VIEW ---------------- */
        return view('staff.index', compact(
            'staffs',
            'salaryStaffs',
            'academic_years',
            'activeAcademicYearId'
        ));
    }



    /* =========================================================
     * STORE
     * ========================================================= */
    public function store(Request $request)
    {
        session(['tab' => 'add_staff']);

        /* ==================================================
            1. HELPER TO SANITIZE NUMBERS
        ================================================== */
        $num = function ($value) {
            return ($value === null || $value === '')
                ? 0
                : (float) str_replace(',', '', $value);
        };

        /* ==================================================
            2. NORMALIZE NUMERIC INPUTS BEFORE VALIDATION
        ================================================== */
        $request->merge([
            'basic'              => $num($request->basic),
            'hra'                => $num($request->hra),
            'da'                 => $num($request->da),
            'ta'                 => $num($request->ta),
            'medical_allowance'  => $num($request->medical_allowance),
            'conveyance'         => $num($request->conveyance),
            'washing_allowance'  => $num($request->washing_allowance),
            'special_allowance'  => $num($request->special_allowance),

            'pf'               => $num($request->pf),
            'esi'              => $num($request->esi),
            'p_tax'            => $num($request->p_tax),
            'tds'              => $num($request->tds),
            'others_deduction' => $num($request->others_deduction),

            'employer_pf'  => $num($request->employer_pf),
            'employer_esi' => $num($request->employer_esi),
        ]);

        /* ==================================================
            3. VALIDATION
        ================================================== */
        $validated = $request->validate([

            /* ---------- STAFF ---------- */
            'name'          => 'required|string|max:255',
            'email'         => 'nullable|email|max:255',
            'phone'         => 'required|digits_between:10,15',
            'address'       => 'nullable|string',
            'category'      => 'required|string|max:100',
            'designation'   => 'required|string|max:100',
            'qualification' => 'required|string|max:255',
            'join_date'     => 'required|date',
            'photo'         => 'nullable|image|max:2048',

            /* ---------- SALARY ---------- */
            'basic' => 'required|numeric|min:0',
            'hra' => 'nullable|numeric|min:0',
            'da' => 'nullable|numeric|min:0',
            'ta' => 'nullable|numeric|min:0',

            'medical_allowance' => 'nullable|numeric|min:0',
            'conveyance'        => 'nullable|numeric|min:0',
            'washing_allowance' => 'nullable|numeric|min:0',
            'special_allowance' => 'nullable|numeric|min:0',

            'pf' => 'nullable|numeric|min:0',
            'esi' => 'nullable|numeric|min:0',
            'p_tax' => 'nullable|numeric|min:0',
            'tds' => 'nullable|numeric|min:0',
            'others_deduction' => 'nullable|numeric|min:0',

            'employer_pf' => 'nullable|numeric|min:0',
            'employer_esi' => 'nullable|numeric|min:0',

            'financial_year_id' => 'required|integer',

            /* ---------- BANK DETAILS (ALL NULLABLE) ---------- */
            'bank_name'       => 'nullable|string|max:255',
            'branch_name'     => 'nullable|string|max:255',
            'account_no'      => 'nullable|string|max:50',
            'account_holder'  => 'nullable|string|max:255',
            'ifsc'            => 'nullable|string|max:20',
        ]);

        /* ==================================================
            4. DATABASE TRANSACTION
        ================================================== */
        DB::transaction(function () use ($validated, $request) {

            /* ---------- PHOTO ---------- */
            $profileImage = $request->file('photo')
                ? $request->file('photo')->store('staff/profile', 'public')
                : null;

            $activeAcademicYearId = AcademicYear::where('status', 1)->value('id');

            /* ---------- STAFF ---------- */
            $staff = Staff::create([
                'name'          => $validated['name'],
                'email'         => $validated['email'] ?? null,
                'phone'         => $validated['phone'],
                'address'       => $validated['address'] ?? null,
                'category'      => $validated['category'],
                'designation'   => $validated['designation'],
                'qualification' => $validated['qualification'],
                'join_date'     => $validated['join_date'],
                'photo'         => $profileImage,
                'status'        => 1,
            ]);

            /* ---------- STAFF ACADEMIC YEAR ---------- */
            StaffAcademicYear::create([
                'staff_id'           => $staff->id,
                'financial_year_id'  => $activeAcademicYearId,
                'status'             => 1,
            ]);

            /* ---------- SALARY CALCULATIONS ---------- */
            $totalEarnings =
                $validated['basic'] +
                $validated['hra'] +
                $validated['da'] +
                $validated['ta'] +
                $validated['medical_allowance'] +
                $validated['conveyance'] +
                $validated['washing_allowance'] +
                $validated['special_allowance'];

            $totalDeductions =
                $validated['pf'] +
                $validated['esi'] +
                $validated['p_tax'] +
                $validated['tds'] +
                $validated['others_deduction'];

            $netSalary = $totalEarnings - $totalDeductions;

            $ctc = $netSalary +
                $validated['employer_pf'] +
                $validated['employer_esi'];

            /* ---------- STAFF SALARY ---------- */
            StaffSalary::create([
                'staff_id'            => $staff->id,
                'basic'               => $validated['basic'],
                'hra'                 => $validated['hra'],
                'da'                  => $validated['da'],
                'ta'                  => $validated['ta'],
                'medical_allowance'   => $validated['medical_allowance'],
                'conveyance'          => $validated['conveyance'],
                'washing_allowance'   => $validated['washing_allowance'],
                'special_allowance'   => $validated['special_allowance'],
                'total_earnings'      => round($totalEarnings, 2),

                'pf'                  => $validated['pf'],
                'esi'                 => $validated['esi'],
                'p_tax'               => $validated['p_tax'],
                'tds'                 => $validated['tds'],
                'others_deduction'    => $validated['others_deduction'],
                'total_deductions'    => round($totalDeductions, 2),

                'net_salary'          => round($netSalary, 2),
                'employer_pf'         => $validated['employer_pf'],
                'employer_esi'        => $validated['employer_esi'],
                'ctc'                 => round($ctc, 2),

                'financial_year_id'   => $validated['financial_year_id'],
                'status'              => 1,
            ]);

            /* ---------- BANK DETAILS (OPTIONAL) ---------- */
            if (
                $validated['bank_name'] ||
                $validated['account_no'] ||
                $validated['ifsc']
            ) {
                $staff->bankDetails()->create([
                    'bank_name'        => $validated['bank_name'] ?? null,
                    'branch_name'      => $validated['branch_name'] ?? null,
                    'acc_no'           => $validated['account_no'] ?? null,
                    'acc_holder_name'  => $validated['account_holder'] ?? null,
                    'ifsc_code'        => $validated['ifsc'] ?? null,
                ]);
            }
        });

        return redirect()
            ->route('staff.index')
            ->with('tab', 'list')
            ->with('success', 'Staff added successfully');
    }



    /* =========================================================
     * EDIT
     * ========================================================= */
    public function edit($id)
    {
        $staff = Staff::with(['salary', 'bankDetails'])->findOrFail($id);
        $academic_years = AcademicYear::where('status', 1)->where('status',1)->get();

        // echo '<pre>';
        // print_r($staff->toArray());die;
        return view('staff.edit', compact('staff','academic_years'));
    }

    /* =========================================================
     * UPDATE
     * ========================================================= */
    public function update(Request $request, $id)
    {
        $staff = Staff::findOrFail($id);

        /* ==================================================
            CASE 1: STATUS TOGGLE ONLY
        ================================================== */
        if ($request->has('status')) {
            $staff->update([
                'status' => $request->boolean('status')
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Staff status updated successfully'
            ]);
        }

        /* ==================================================
            COMMON MONEY SANITIZER (SAME AS STORE)
        ================================================== */
        $num = function ($value) {
            return ($value === null || $value === '')
                ? 0
                : (float) str_replace(',', '', $value);
        };

        // Normalize numeric inputs BEFORE validation
        $request->merge([
            'basic'              => $num($request->basic),
            'hra'                => $num($request->hra),
            'da'                 => $num($request->da),
            'ta'                 => $num($request->ta),
            'medical_allowance'  => $num($request->medical_allowance),
            'conveyance'         => $num($request->conveyance),
            'washing_allowance'  => $num($request->washing_allowance),
            'special_allowance'  => $num($request->special_allowance),

            'pf'               => $num($request->pf),
            'esi'              => $num($request->esi),
            'p_tax'            => $num($request->p_tax),
            'tds'              => $num($request->tds),
            'others_deduction' => $num($request->others_deduction),

            'employer_pf'  => $num($request->employer_pf),
            'employer_esi' => $num($request->employer_esi),
        ]);

        /* ==================================================
            VALIDATION
        ================================================== */
        $validated = $request->validate([
            // STAFF
            'name' => 'required|string|max:255',
            'email' => 'nullable|email|max:255',
            'phone' => 'required|digits_between:10,15',
            'address' => 'nullable|string',
            'category' => 'required|string',
            'designation' => 'required|string',
            'qualification' => 'required|string',
            'join_date' => 'required|date',
            'photo' => 'nullable|image|max:2048',

            // SALARY
            'basic' => 'required|numeric|min:0',
            'hra' => 'nullable|numeric|min:0',
            'da' => 'nullable|numeric|min:0',
            'ta' => 'nullable|numeric|min:0',
            'medical_allowance' => 'nullable|numeric|min:0',
            'conveyance' => 'nullable|numeric|min:0',
            'washing_allowance' => 'nullable|numeric|min:0',
            'special_allowance' => 'nullable|numeric|min:0',

            'pf' => 'nullable|numeric|min:0',
            'esi' => 'nullable|numeric|min:0',
            'p_tax' => 'nullable|numeric|min:0',
            'tds' => 'nullable|numeric|min:0',
            'others_deduction' => 'nullable|numeric|min:0',

            'employer_pf' => 'nullable|numeric|min:0',
            'employer_esi' => 'nullable|numeric|min:0',

            // BANK
            'bank_name' => 'nullable|string|max:255',
            'branch_name' => 'nullable|string|max:255',
            'account_no' => 'nullable|string|max:30',
            'account_holder' => 'nullable|string|max:255',
            'ifsc' => 'nullable|string|max:20',
        ]);

        DB::transaction(function () use ($request, $validated, $staff) {

            /* ---------- PHOTO ---------- */
            if ($request->hasFile('photo')) {
                $staff->photo = $request->file('photo')
                    ->store('staff/profile', 'public');
                $staff->save();
            }

            /* ---------- STAFF UPDATE ---------- */
            $staff->update([
                'name' => $validated['name'],
                'email' => $validated['email'] ?? null,
                'phone' => $validated['phone'],
                'address' => $validated['address'] ?? null,
                'category' => $validated['category'],
                'designation' => $validated['designation'],
                'qualification' => $validated['qualification'],
                'join_date' => $validated['join_date'],
            ]);

            /* ---------- SALARY CALC (SAFE) ---------- */
            $totalEarnings =
                $validated['basic'] +
                $validated['hra'] +
                $validated['da'] +
                $validated['ta'] +
                $validated['medical_allowance'] +
                $validated['conveyance'] +
                $validated['washing_allowance'] +
                $validated['special_allowance'];

            $totalDeductions =
                $validated['pf'] +
                $validated['esi'] +
                $validated['p_tax'] +
                $validated['tds'] +
                $validated['others_deduction'];

            $netSalary = $totalEarnings - $totalDeductions;

            $ctc =
                $netSalary +
                $validated['employer_pf'] +
                $validated['employer_esi'];

            /* ---------- SALARY UPDATE ---------- */
            $staff->salary()->update([
                'basic' => $validated['basic'],
                'hra' => $validated['hra'],
                'da' => $validated['da'],
                'ta' => $validated['ta'],
                'medical_allowance' => $validated['medical_allowance'],
                'conveyance' => $validated['conveyance'],
                'washing_allowance' => $validated['washing_allowance'],
                'special_allowance' => $validated['special_allowance'],
                'total_earnings' => round($totalEarnings, 2),

                'pf' => $validated['pf'],
                'esi' => $validated['esi'],
                'p_tax' => $validated['p_tax'],
                'tds' => $validated['tds'],
                'others_deduction' => $validated['others_deduction'],
                'total_deductions' => round($totalDeductions, 2),

                'net_salary' => round($netSalary, 2),
                'employer_pf' => $validated['employer_pf'],
                'employer_esi' => $validated['employer_esi'],
                'ctc' => round($ctc, 2),
            ]);

            /* ---------- BANK UPDATE ---------- */
            if ($staff->bankDetails) {
                $staff->bankDetails->update([
                    'bank_name' => $validated['bank_name'] ?? null,
                    'branch_name' => $validated['branch_name'] ?? null,
                    'acc_no' => $validated['account_no'] ?? null,
                    'acc_holder_name' => $validated['account_holder'] ?? null,
                    'ifsc_code' => $validated['ifsc'] ?? null,
                ]);
            }else {
                $staff->bankDetails()->create([
                    'bank_name'        => $validated['bank_name'] ?? null,
                    'branch_name'      => $validated['branch_name'] ?? null,
                    'acc_no'           => $validated['account_no'] ?? null,
                    'acc_holder_name'  => $validated['account_holder'] ?? null,
                    'ifsc_code'        => $validated['ifsc'] ?? null,
                ]);
            }
        });

        return redirect()
            ->route('staff.index')
            ->with('success', 'Staff updated successfully');
    }


    public function show($id)
    {
        $staff = Staff::with([
            'salary.financialYear',
            'bankDetails',
            'academicYears.financialYear'
        ])->findOrFail($id);

        return view('staff.show', compact('staff'));
    }



    /* =========================================================
     * DELETE (Soft logic via academic year)
     * ========================================================= */
    public function destroy($id)
    {
        $activeAcademicYearId = AcademicYear::where('status', 1)->value('id');

        StaffAcademicYear::where('staff_id', $id)
            ->where('financial_year_id', $activeAcademicYearId)
            ->update(['status' => 0]);

        return redirect()->route('staff.index')->with('success', 'Staff removed from current academic year');
    }




    public function paySalary(Request $request)
    {
        $request->validate([
            'staff_id' => 'required|integer',
            'financial_year_id' => 'required|integer',
            'month' => 'required|date_format:Y-m', // 👈 IMPORTANT
        ]);

        DB::transaction(function () use ($request) {

            //  Prevent duplicate payment
            $alreadyPaid = StaffMonthlySalary::where('staff_id', $request->staff_id)
                ->where('pay_for_month', $request->month)
                ->where('financial_year_id', $request->financial_year_id)
                ->exists();

            if ($alreadyPaid) {
                throw new \Exception('Salary already paid for this month');
            }

            $salary = StaffSalary::where('staff_id', $request->staff_id)
                ->where('financial_year_id', $request->financial_year_id)
                ->where('status', 1)
                ->firstOrFail();

            StaffMonthlySalary::create([
                'staff_id' => $salary->staff_id,

                // earnings
                'basic' => $salary->basic,
                'hra' => $salary->hra,
                'medical_allowance' => $salary->medical_allowance,
                'conveyance' => $salary->conveyance,
                'washing_allowance' => $salary->washing_allowance,
                'special_allowance' => $salary->special_allowance,
                'total_earnings' => $salary->total_earnings,

                // deductions
                'pf' => $salary->pf,
                'esi' => $salary->esi,
                'p_tax' => $salary->p_tax,
                'tds' => $salary->tds,
                'advance' => 0,
                'others_deduction' => $salary->others_deduction,
                'total_deductions' => $salary->total_deductions,

                // net
                'net_salary' => $salary->net_salary,
                'employer_pf' => $salary->employer_pf,
                'employer_esi' => $salary->employer_esi,
                'ctc' => $salary->ctc,

                // ✅ FIXED HERE
                'pay_for_month' => $request->month, // e.g. 2025-11
                'paid_date' => now(),
                'financial_year_id' => $request->financial_year_id,
            ]);
        });

        return back()
            ->with('tab', 'staff_salary')
            ->with('success', 'Salary paid successfully');
    }


    public function salarySlip($staffId, $month)
    {
        $slip = StaffMonthlySalary::with(['staff', 'financialYear'])
            ->where('staff_id', $staffId)
            ->where('pay_for_month', $month)
            ->firstOrFail();

        return view('staff.salary-slip', compact('slip'));
    }

    public function payAllPendingSalaries(Request $request)
    {
        $month = $request->month;
        $financialYearId = $request->financial_year_id;

        $pendingStaffs = Staff::with(['salary' => function($q) use ($financialYearId) {
                $q->where('financial_year_id', $financialYearId)
                ->where('status', 1);
            },
            'monthlySalaries' => function($q) use ($month, $financialYearId) {
                $q->where('pay_for_month', $month)
                ->where('financial_year_id', $financialYearId);
            }
        ])
        ->whereHas('salary', function($q) use ($financialYearId) {
            $q->where('financial_year_id', $financialYearId)
            ->where('status', 1);
        })
        ->get()
        ->filter(function($staff) {
            return $staff->monthlySalaries->isEmpty(); // only pending
        });

        DB::transaction(function() use ($pendingStaffs, $month, $financialYearId) {
            foreach($pendingStaffs as $staff) {
                $salary = $staff->salary->first();
                if ($salary) {
                    \App\Models\StaffMonthlySalary::create([
                        'staff_id' => $staff->id,

                        // Earnings
                        'basic' => $salary->basic,
                        'hra' => $salary->hra,
                        'medical_allowance' => $salary->medical_allowance,
                        'conveyance' => $salary->conveyance,
                        'washing_allowance' => $salary->washing_allowance,
                        'special_allowance' => $salary->special_allowance,
                        'total_earnings' => $salary->total_earnings,

                        // Deductions
                        'pf' => $salary->pf,
                        'esi' => $salary->esi,
                        'p_tax' => $salary->p_tax,
                        'tds' => $salary->tds,
                        'advance' => 0,
                        'others_deduction' => $salary->others_deduction,
                        'total_deductions' => $salary->total_deductions,

                        // Net & Employer
                        'net_salary' => $salary->net_salary,
                        'employer_pf' => $salary->employer_pf,
                        'employer_esi' => $salary->employer_esi,
                        'ctc' => $salary->ctc,

                        'pay_for_month' => $month,
                        'paid_date' => now(),
                        'financial_year_id' => $financialYearId,
                    ]);
                }
            }
        });

        return back()->with('success', 'All pending salaries have been paid successfully')
        ->with('tab', 'staff_salary');
    }

    private function num($value): float
    {
        return (float) ($value ?? 0);
    }



}
