<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Models\School;
use App\Models\SchoolSession;
use App\Models\AcademicYear;

class SchoolSetupController extends Controller
{
    public function index()
    {
        $school = School::first();
        $academic_years = AcademicYear::get();
        $sessions = SchoolSession::get();

        return view('school.setup', compact('school', 'academic_years','sessions'));
    }

    public function saveSchoolDetails(Request $request)
    {
        $request->validate([
            'name' => 'required',
            'established_year' => 'required|numeric',
            'address' => 'required',
            'phone' => 'required',
            'email' => 'required|email',
            'logo' => 'nullable|image'
        ]);

        $school = School::first() ?? new School();

        $school->fill($request->except('logo'));

        if ($request->hasFile('logo')) {
            $logoPath = $request->logo->store('school', 'public');
            $school->logo = $logoPath;
        }

        $school->save();

        return back()->with('success', 'School details saved successfully.');
    }

    public function addYear(Request $request)
    {
        $request->validate([
            'label' => 'required|unique:financial_years,label',
            'year_from' => 'required|unique:financial_years,year_from',
            'year_to' => 'required|unique:financial_years,year_to'
        ]);

        DB::transaction(function () use ($request) {
            AcademicYear::where('status', 1)->update(['status' => 0]);

            AcademicYear::create([
                'label'     => $request->label,
                'year_from' => $request->year_from,
                'year_to'   => $request->year_to
            ]);
        });


        return back()->with('success', 'Academic Year added successfully.');
    }

    public function updateYear(Request $request, $id)
    {
        $year = AcademicYear::findOrFail($id);
        if($request->has('status')){
            DB::transaction(function () use ($request, $year) {

                // Count active years
                $activeCount = AcademicYear::where('status', 1)->count();

                // ❌ Prevent deactivating the last active year
                if ($year->status == 1 && $request->status == 0 && $activeCount <= 1) {
                    throw new \Exception('At least one academic year must remain active.');
                }

                // ✅ If activating this year → deactivate others
                if ($request->status == 1) {
                    AcademicYear::where('id', '!=', $year->id)
                        ->update(['status' => 0]);
                }

                // Update current year
                $year->update([
                    'status' => $request->status
                ]);
            });
        } else {

            $request->validate([
                'label' => 'required|string|max:255|unique:financial_years,label,' . $id,
                'year_from' => 'required|unique:financial_years,year_from,' . $id,
                'year_to' => 'required|unique:financial_years,year_to,' . $id,
            ]);

            $year->update([
                'label' => $request->label,
                'year_from' => $request->year_from,
                'year_to' => $request->year_to,
            ]);
        }

        return response()->json(['success' => true]);
    }


    public function deleteYear($id)
    {
        AcademicYear::findOrFail($id)->delete();

        return response()->json(['success' => true]);
    }

    public function addSession(Request $request)
    {
        $request->validate([
            'session_name' => 'required|unique:sessions,name'
        ]);

        SchoolSession::create([
            'name' => $request->session_name
        ]);

        return back()->with('success', 'Session added successfully.');
    }


    public function updateSession(Request $request, $id)
    {
        $session = SchoolSession::findOrFail($id);

        if($request->has('status')){
            $session->update([
                'status' => $request->status
            ]);
        } else {

            $request->validate([
                'session_name' => 'required|string|max:255|unique:sessions,name,' . $id,
            ]);

            $session->update([
                'name' => $request->session_name,
            ]);
        }

        return response()->json(['success' => true]);
    }


    public function deleteSession($id)
    {
        SchoolSession::findOrFail($id)->delete();

        return response()->json(['success' => true]);
    }

}
